(ns codescene.features.pm-data.trello.trello-provider
  "A PM Data Provider for fetching data from Trello.
   Uses a cache for fetching data. 
   This means that a single instance of the Provider will only ever fetch data once,
   and a new instance (with a new cache) is needed for fetching updated data."
  (:require [codescene.pm-data.pm-data-provider :refer [PmDataProvider]]
            [codescene.features.pm-data.pm-data-configuration :refer [PmDataConfiguration]]
            [codescene.features.pm-data.pm-data-provider :refer [-create-pm-data-provider]]
            [codescene.features.pm-data.trello.trello-fetcher :as f]
            [codescene.features.pm-data.trello.trello-cache :as tc]
            [codescene.features.pm-data.trello.trello-ticket-id-extractor :as x]
            [codescene.pm-data.provider-common :as c]
            [taoensso.timbre :as log]))

(defn fetch-project
  [{:keys [api-client] :as provider-def} project since]
  (let [issues (->> (tc/fetch-issues since api-client project provider-def)
                    ;; Note that href is included in fetched data, contrary to other providers
                    (c/with-project-id :id project))]
    (log/infof "Import %d issues from Trello project %s" (count issues) project)
    issues))

(defn- get-pm-data [provider-def {:keys [since] :as _pm-data-context}]
  (let [{:keys [external-project-ids]} provider-def
        issues (->> external-project-ids
                    (mapcat #(fetch-project provider-def % since))
                    (into []))]
    {:issues issues
     :provider-def provider-def}))

(defn- get-configuration-data [{:keys [api-client] :as _provider-def}]
  (log/info "Fetching configuration data from Trello")
  (let [boards      (f/fetch-boards api-client)
        board-names (map first boards)
        board-ids   (map second boards)]
    {:work-types  (->> (f/fetch-labels api-client board-ids)
                       (mapv c/->name-and-key))
     :transitions (->> (f/fetch-lists api-client board-ids)
                       (mapv c/->name-and-key))
     :projects    (mapv c/->name-and-key board-names)}))

(defn- validate-settings
  [{:keys [api-client] :as _provider-def}]
  (f/fetch-me api-client)
  nil)

(defrecord TrelloProvider [provider-def pm-data-context]
  PmDataProvider
  (-make-ticket-id-extractor [_this _repo] (x/make-ticket-id-extractor provider-def))
  (-get-pm-data [_this] (get-pm-data provider-def pm-data-context))
  PmDataConfiguration
  (-get-configuration-data [_this] (get-configuration-data provider-def))
  (-validate-settings [_this] (validate-settings provider-def)))

(defmethod -create-pm-data-provider "trello"
  [provider-def pm-data-context]
  (->TrelloProvider provider-def pm-data-context))

(comment
  (def since (evolutionary-metrics.trends.dates/string->date "2021-02-25"))
  (def context {:since since})
  (def api-client (codescene.features.pm-data.trello.trello-api/trello-auth
                   (System/getenv "TRELLO_API_KEY")
                   (System/getenv "TRELLO_API_TOKEN")))
  (def provider-def {:http-timeout 10000
                     :external-project-ids ["Empear-On-Prem"]
                     :supported-work-types ["Bug" "Feature" "Improvement/Refactor" "Documentation"]
                     :defect-and-failure-labels "Bug"
                     :rename-work-types [["Refactoring" "R"] ["Bug" "B"]]
                     :work-in-progress-transition-names ["In Progress"]
                     :api-client api-client})
  (get-configuration-data provider-def)
  (get-pm-data provider-def context))