(ns codescene.features.pm-data.shortcut.shortcut-provider
  "A PM Data Provider for fetching data from Shortcut.
   Uses cache for fetching data. 
   This means that a single instance of the Provider will only ever fetch data once,
   and a new instance (with a new cache) is needed for fetching updated data."
  (:require [codescene.pm-data.pm-data-provider :refer [PmDataProvider]]
            [codescene.features.pm-data.pm-data-configuration :refer [PmDataConfiguration]]
            [codescene.features.pm-data.pm-data-provider :refer [-create-pm-data-provider]]
            [codescene.features.pm-data.shortcut.shortcut-fetcher :as f]
            [codescene.features.pm-data.shortcut.cache :as cache]
            [codescene.features.pm-data.shortcut.shortcut-ticket-id-extractor :as x]
            [taoensso.timbre :as log]))

(defn- fetch-project [{:keys [api-client] :as _provider-def} project since]
  (let [issues (cache/fetch-issues since api-client project)]
    (log/infof "Import %d issues from Shortcut project %s" (count issues) project)
    issues))

(defn- get-pm-data
  [provider-def {:keys [since] :as _pm-data-context}]
  (let [{:keys [external-project-ids]} provider-def
        issues (->> external-project-ids
                    (mapcat #(fetch-project provider-def % since))
                    (into []))]
    {:issues       issues
     :provider-def {:cost-unit "issues"
                    :work-in-progress-transition-names ["started"]
                    :work-done-transition-names ["completed"]
                    :defect-and-failure-labels ["bug"]}}))

(defn- get-configuration-data [{:keys [api-client] :as _provider-def}]
  (log/info "Fetching configuration data from Shortcut")
  {:projects (f/fetch-projects api-client)})

(defn- validate-settings
  [{:keys [api-client] :as _provider-def}]
  (f/fetch-member api-client)
  nil)

(deftype ShortcutProvider [provider-def pm-data-context]
  PmDataProvider
  (-make-ticket-id-extractor [_this _repo] (x/make-ticket-id-extractor))
  (-get-pm-data [_this] (get-pm-data provider-def pm-data-context))
  PmDataConfiguration
  (-get-configuration-data [_this] (get-configuration-data provider-def))
  (-validate-settings [_this] (validate-settings provider-def)))

(defmethod -create-pm-data-provider "shortcut"
  [provider-def pm-data-context]
  (->ShortcutProvider provider-def pm-data-context))

(comment
  (def api-client (codescene.features.pm-data.shortcut.shortcut-api/shortcut-auth (System/getenv "SHORTCUT_TOKEN")))
  (def since (evolutionary-metrics.trends.dates/string->date "2022-02-10"))
  (def context {:since since})
  (get-configuration-data {:api-client api-client})
  (validate-settings {:api-client api-client})
  (get-pm-data {:api-client api-client
                :external-project-ids ["API"]}
               context)
  )
