(ns codescene.features.factors.results
  (:require
   [camel-snake-kebab.core :as csk]
   [clj-time.core :as tc]
   [codescene.analysis.author-metrics :as author-metrics]
   [codescene.analysis.vcs-metrics :as vcs-metrics]
   [codescene.analysis.paths :as paths]
   [codescene.biomarkers.team-level-hotspots :as team-level-hotspots]
   [codescene.factors.code-health.presentation :as code-health-presentation]
   [codescene.factors.core :as factors]
   [codescene.factors.delivery.presentation :as delivery-presentation]
   [codescene.factors.knowledge.presentation :as knowledge-presentation]
   [codescene.factors.team-code.presentation :as tca-presentation]
   [codescene.features.dashboard.core :as dashboard-common]
   [codescene.features.presentation.time-scope :as scope]
   [codescene.features.factors.kpi-time-series :as kpi-time-series]
   [codescene.util.json :as json]
   [codescene.util.time :as ut]
   [medley.core :as m]
   [ring.util.response :as ring-response]
   [taoensso.tufte :as tufte :refer [defnp]])
  (:import (java.time Instant)))

(defn- teams-for
  [analysis-dir]
  (team-level-hotspots/teams-for {:analysis-path-fn (partial paths/make-analysis-path-to analysis-dir)}))

(defn- bisect-left
  "Left-bisects the sample array on :datetime key.
  Assumes sample array is sorted ascending on :datetime.
  Return an array of two objects, the ones surrounding the date parameter.
  Inspired by https://observablehq.com/@d3/d3-bisect"
  [date samples]
  (let [[befores afters] (split-with #(tc/after? date (:datetime %)) samples)]
    [(last befores) (first afters)]))

(defn- time-fract
  "Calculates the relationship between time intervals: (before -- at)/(before -- after).
  Returns a rational number or 0 or 1
  Expects that `before < at < after"
  [before at after]
  (let [before->at (tc/interval before at)
        at->after (tc/interval before after)
        first-interval (tc/in-millis before->at)
        last-interval (tc/in-millis at->after)]
    (/ first-interval last-interval)))

(defn- interpolate-number
  "Interpolates two numbers a and b, with the fractional component t.
  Inspired by https://observablehq.com/@d3/d3-interpolatenumber
  The return value is always a double to avoid confusion."
  [a b t]
  (let [a-fract (* a (- 1 t))
        b-fract (* b t)]
    (double (+ a-fract b-fract))))

(defn- bisect-interpolate [date samples]
  (let [[before after] (bisect-left date samples)]
    (cond
      (tc/equal? date (:datetime before)) (:kpi before)
      (tc/equal? date (:datetime after)) (:kpi after)
      :else (let [a (:kpi before)
                  b (:kpi after)
                  t (time-fract (:datetime before) date (:datetime after))
                  res (interpolate-number a b t)]
              (double res)))))

(defn get-interpolated-value-at-date
  "Takes a date to get interpolated value from,
  and a samples collecion, containing :datetime and :kpi,
  assumed to be sorted on date in ascending order.

  Returns an interpolated value for the date,
  based on the values of the samples collection."
  [date samples]
  (cond
    (tc/before? date (:datetime (first samples))) (:kpi (first samples))
    (tc/equal? date (:datetime (first samples))) (:kpi (first samples))
    (tc/after? date (:datetime (last samples))) (:kpi (last samples))
    :else (bisect-interpolate date samples)))

(defn- delivery-kpi->weekly-avg-development-time [{:keys [date] :as delivery-performance}]
  (let [weekly-sub-kpis (-> delivery-performance :kpis :week :sub-kpis)
        development-time (first (filter (fn [sub-kpi] (= (:key sub-kpi) "development-time")) weekly-sub-kpis))]
    (merge
     (select-keys development-time [:value-in-minutes :name :completed-tasks :precision])
     {:date date})))

(defn weekly-avg-development-times
  "Returns the average weekly development times in reverse chronological order for neat presentation in a table."
  [absolute-analysis-path]
  (let [delivery-kpi-time-series (factors/time-series-by-type absolute-analysis-path delivery-presentation/kpi-key)]
    (->> delivery-kpi-time-series
         (map delivery-kpi->weekly-avg-development-time)
         (remove (fn [res] (nil? (:value-in-minutes res))))
         (sort-by :date)
         reverse)))

(defn json-response [body]
  (-> body
      ring-response/response
      ;; TODO @aasa if we can wrap the response types for all routes in onprem
      ;; as well as in cloud, we can skip explicitly setting the content type here
      (ring-response/content-type "application-json")))

(defn ->json-string [factors]
  (json/generate-string factors {:key-fn #(csk/->camelCase (name %))}))

(def ^:private default-sample-value 0)

(defn- value-sample
  "Value defaults to 0 if the sample contains an error or info, otherwise the value is
  determined by the fetch-value-fn.

  A date is usually required for each sample, but when included as a child of a sample
  that already has a date it is redundant, and we use this variant instead. Typical
  cases are samples under the :teams key, or delivery samples under :kpis/:week etc."
  [sample fetch-value-fn optional-keys]
  (let [initial-value (m/assoc-some {:value default-sample-value} :value (fetch-value-fn sample))]
    (->> optional-keys
         (map (fn [key] (m/assoc-some initial-value key (key sample))))
         (reduce merge))))

(defn- date-value-sample
  "Create a renderable sample with required date. A sample is considered to be
  renderable if it has a date and a value. The samples for different kpis are
  slightly different in other regards, but this is the least common denominator."
  [{:keys [date] :as sample} fetch-value-fn optional-keys]
  (if (nil? date)
    (throw (ex-info "Sample does not contain a date" {:sample sample}))
    (merge (value-sample sample fetch-value-fn optional-keys)
           {:date date})))

(defn- avg-code-health-value-from-ch-sample [{:keys [sub-kpis]}]
  (:value (m/find-first (fn [sub-kpi] (= (name code-health-presentation/average-code-health-key) (:key sub-kpi))) sub-kpis)))

(defn- team-sample-with-description
  [todays-descriptions
   {:keys [team kpi] :as team-code-health-kpi-sample}]
  (m/assoc-some (value-sample team-code-health-kpi-sample avg-code-health-value-from-ch-sample [:sub-kpis :team :info :error])
                :weighted-kpi kpi
                :descriptions (:descriptions (m/find-first #(= team (:team %)) (:teams todays-descriptions)))))

(defn- goals-sample [{:keys [kpi] :as goals-sample}]
  (m/assoc-some (value-sample goals-sample avg-code-health-value-from-ch-sample [:sub-kpis :info :error])
                :weighted-kpi kpi))

(defn- code-health-sample->renderable
  "Turn a code health sample into a trend graph renderable sample."
  [descriptions-lookup
   {:keys [kpi date teams goals] :as code-health-kpi-sample}]
  (let [todays-descriptions (descriptions-lookup date)]
    (m/assoc-some (date-value-sample code-health-kpi-sample avg-code-health-value-from-ch-sample [:sub-kpis :info :error])
                  :weighted-kpi kpi
                  :teams (map (partial team-sample-with-description todays-descriptions) teams)
                  :goals (when goals (goals-sample goals))
                  :descriptions (:descriptions todays-descriptions))))

(defn- fast-trend->descriptions-lookup [code-health-fast-trend]
  (into {} (map (juxt :date identity) code-health-fast-trend)))

(defn- code-health-trend->renderable [code-health-fast-trend code-health-trend]
  (let [descriptions-lookup (fast-trend->descriptions-lookup code-health-fast-trend)]
    (map (partial code-health-sample->renderable descriptions-lookup) code-health-trend)))

(defn- knowledge-sample->renderable [sample]
  (date-value-sample sample #(:kpi %) [:sub-kpis :descriptions :error :info]))

(defn knowledge-trend->renderable [knowledge-trend]
  (map knowledge-sample->renderable knowledge-trend))

(defn- tca-teams->renderable [sample]
  (value-sample sample #(:kpi %) [:sub-kpis :descriptions :team :error :info]))

(defn- tca-sample->renderable [{:keys [teams] :as sample}]
  (m/assoc-some
   (date-value-sample sample #(:kpi %) [:sub-kpis :descriptions :error :info])
   :teams (map tca-teams->renderable teams)))

(defn tca-trend->renderable [tca-trend]
  (map tca-sample->renderable tca-trend))

(defn- delivery-kpi-sample->renderable [sample]
  (value-sample sample #(:kpi %) [:sub-kpis :descriptions :precision :error :info]))

(defn- delivery-kpis-sample->renderable
  [{:keys [day week month year]}]
  {:day (delivery-kpi-sample->renderable day)
   :week (delivery-kpi-sample->renderable week)
   :month (delivery-kpi-sample->renderable month)
   :year (delivery-kpi-sample->renderable year)})

(defn- delivery-teams->renderable [{:keys [kpis team]}]
  (m/assoc-some {}
                :kpis (delivery-kpis-sample->renderable kpis)
                :team team))

(defn- delivery-sample->renderable
  [{:keys [date kpis teams error]}]
  (m/assoc-some
   {:date date}
   :kpis (when (some? kpis) (delivery-kpis-sample->renderable kpis))
   :teams (when (some? teams) (map delivery-teams->renderable teams))
   :error error))

(defn- delivery-trend->renderable
  [delivery-trend]
  (map delivery-sample->renderable delivery-trend))

(defnp ->renderable-trends
  ([absolute-analysis-path beginning end]
   (->renderable-trends absolute-analysis-path beginning end false))
  ([absolute-analysis-path beginning end trailing-sample?]
   (let [code-health-fast-trend (factors/time-series-by-type absolute-analysis-path code-health-presentation/fast-trend-key beginning end trailing-sample?)
         code-health-trend (factors/time-series-by-type absolute-analysis-path code-health-presentation/kpi-key beginning end trailing-sample?)
         renderable-code-health-trend (code-health-trend->renderable code-health-fast-trend code-health-trend)
         delivery-trend (factors/time-series-by-type absolute-analysis-path delivery-presentation/kpi-key beginning end trailing-sample?)
         knowledge-trend (factors/time-series-by-type absolute-analysis-path knowledge-presentation/kpi-key beginning end trailing-sample?)
         tca-trend (factors/time-series-by-type absolute-analysis-path tca-presentation/kpi-key beginning end trailing-sample?)]
     {code-health-presentation/kpi-key renderable-code-health-trend
      knowledge-presentation/kpi-key (knowledge-trend->renderable knowledge-trend)
      tca-presentation/kpi-key (tca-trend->renderable tca-trend)
      delivery-presentation/kpi-key (delivery-trend->renderable delivery-trend)})))


(defn trends->renderable
  ;; TODO: This function will replace ->renderable-trends when we no longer fetch trends from the analysis results.
  "Applies the rendering functions to the trends fetched from the database."
  [{:keys [code-health-fast-trend code-health delivery-performance knowledge-distribution team-code-alignment]} beginning end trailing-sample?]
  (let [truncate (partial factors/sort-and-truncate-kpi-trend beginning end trailing-sample?)]
    {code-health-presentation/kpi-key (code-health-trend->renderable (truncate code-health-fast-trend) (truncate code-health))
     knowledge-presentation/kpi-key (knowledge-trend->renderable (truncate knowledge-distribution))
     tca-presentation/kpi-key (tca-trend->renderable (truncate team-code-alignment))
     delivery-presentation/kpi-key (delivery-trend->renderable (truncate delivery-performance))}))


(defn end-date
  "Tries to figure out the appropriate analysis window, based on files that may or may not exist in older analyses.
  If none of our date files exist, we just use the current time. Then atleast the dashboard does not immedately die,
  even tho it will not show pretty graphs."
  [absolute-analysis-path]
  (let [{:keys [age-time-now time-now]} (json/read-when-exists (str absolute-analysis-path "/" paths/run-summary-json))]
    (cond
     ;; Preferred and up-to-date way of pinning the end of the analysis window
      (string? age-time-now) (ut/instant age-time-now)
     ;; Older version, more amiguous due to the logic of determining `now`
      (string? time-now) (ut/instant time-now)
     ;; Very old analyses?
      :else (Instant/now))))

(defn- keep-selected-delivery-properties
  [delivery-kpi delivery-details]
  (-> delivery-kpi
      (update-in [:kpis :day] select-keys delivery-details)
      (update-in [:kpis :week] select-keys delivery-details)
      (update-in [:kpis :month] select-keys delivery-details)
      (update-in [:kpis :year] select-keys delivery-details)))

(defn- remove-sub-kpi-name
  [kpi-sample]
  (m/update-existing kpi-sample :sub-kpis (fn [sub-kpis]
                                            (map #(dissoc % :name) sub-kpis))))

(defn remove-error-data-points
  "Remove all data points that are not valid.
  To display misconfiguration we instead use the config diagnostic object, rather than
  by reading error data points."
  [key value]
  [key (remove #(contains? % :error) value)])

(defn- split-trend-at-last-element [k trends]
  (let [trend (get trends k)
        [other-samples last-sample] (split-at (dec (count trend)) trend)]
    {k {:other-samples other-samples
        :last-sample last-sample}}))

(defn keep-properties [props-to-keep delivery-props-to-keep]
  (fn [{:keys [kpis] :as kpi-sample}]
    (cond->> (select-keys kpi-sample props-to-keep)
      kpis (#(keep-selected-delivery-properties % delivery-props-to-keep)))))

(defn- prepare-samples-for-projects-page
  "Removes data not used by the UI for all samples except the last one. Removing the sub-kpi property
  saves quite a lot of data from being sent to the client."
  [{:keys [other-samples last-sample]}]
  (let [clean-others (->> other-samples
                          (map (keep-properties [:date :value :kpis :sampled-at] [:value :error :info]))
                          (remove #(contains? % :error)))
        clean-last (->> last-sample
                        (map (keep-properties
                               [:date :value :weighted-kpi :kpis :sampled-at :info :error :sub-kpis]
                               [:value :error :info :sub-kpis]))
                        (map remove-sub-kpi-name))]
    (concat clean-others clean-last)))

(defn- light-trends-for-projects-page
  "Light trends for the projects page. We only need complete data in the last sample for each trend."
  [trends]
  (let [split-trends (->> factors/factor-keys
                          (map #(split-trend-at-last-element % trends))
                          (into {}))]
      (m/map-vals prepare-samples-for-projects-page split-trends)))


(defnp projects-page-time-series [absolute-analysis-path]
  (let [[beginning end] (scope/interval-from-scope (end-date absolute-analysis-path) "month")
        ;; TODO: `renderable-trends` is slowest 
        {:keys [version] :as renderable-trends} (->renderable-trends absolute-analysis-path beginning end true)
        factor-series-only (select-keys renderable-trends factors/factor-keys)]
    (-> (light-trends-for-projects-page factor-series-only)
        (assoc  :version version))))


(defn projects-page-time-series-from-db
  [db-spec {:keys [analysis-id project-id end-time]}]
  (let [end-time-instant (Instant/ofEpochMilli (.getMillis end-time))
        [beginning end] (scope/interval-from-scope end-time-instant "month")]
    (-> (kpi-time-series/time-series-by-analysis-id db-spec project-id analysis-id)
        (trends->renderable beginning end true)
        (select-keys  factors/factor-keys)
        light-trends-for-projects-page
        ;; TODO: use a real version scheme if necessary
        (assoc :version 0))))


(comment
  (->> (projects-page-time-series "/Users/jlindbergh/projects/codescene/codescene/features/test/codescene/features/factors/results_test_data/good_files_from_codescene_staging")
       :code-health))
;; -- END projects page

;; Dashboard & trend graphs
(defn factor-data-for
  ([absolute-analysis-path]
   ;; All the data, made renderable in the backend
   (let [beginning (ut/instant "1970-01-01")
         end (end-date absolute-analysis-path)]
     (factor-data-for absolute-analysis-path beginning end)))
  ;; Data from beginning to end instant
  ([absolute-analysis-path beginning end]
   (-> (->renderable-trends absolute-analysis-path beginning end)
       (assoc :teams (teams-for absolute-analysis-path))
       (assoc :file-summary (dashboard-common/file-summary-per-language absolute-analysis-path))
       (assoc :dashboard (dashboard-common/dashboard-data absolute-analysis-path)))))


(defn period-statistics
  [analysis-path-fn beginning end]
  (let [contributors (author-metrics/contributors-in-interval {:analysis-path-fn analysis-path-fn} beginning end)
        commits (vcs-metrics/revisions-in-interval {:analysis-path-fn analysis-path-fn} beginning end)
        prs (vcs-metrics/prs-by-interval {:analysis-path-fn analysis-path-fn} beginning end)
        total-negative-findings (reduce + (map #(or (:total-negative-findings %) 0) prs))
        ignored-findings (reduce + (map #(or (:ignored-findings %) 0) prs))
        suppressed-findings (reduce + (map #(or (:suppressed-findings %) 0) prs))]
    {:contributors (count contributors)
     :commits commits
     :pull-requests (count prs)
     :negative-findings total-negative-findings
     :findings-fixed (- total-negative-findings ignored-findings suppressed-findings)
     :findings-ignored ignored-findings
     :findings-suppressed suppressed-findings}))

(comment
 (let [path-fn (partial codescene.analysis.paths/make-analysis-path-to
                        #_"/home/ulrikawiss/Downloads/58-trend-graph-fix/33464/analysis20220915061601"
                        #_"/home/ulrikawiss/Downloads/58-trend-graph-fix/34658/analysis20230502111543"
                        "/home/ulrikawiss/Downloads/58-trend-graph-fix/20819/analysis20230524024601"
                        #_"/home/ulrikawiss/Downloads/churntrendinvestig/results (4)/analysis20230508024601")
       absolute-analysis-path (path-fn "")
       end-datee (end-date absolute-analysis-path)
       [monthago end] (scope/interval-from-scope end-datee "month")]
   (period-statistics path-fn monthago end)))
