(ns codescene.features.code-coverage.gates.code-lines-detector
  "A code coverage report only contains info about the code that was executed.
   That's not good enough: our gates need to be able to detect application code without any coverage, too.
   However, just counting lines won't fly: it will lead to many false negatives.
   Instead, we need to detect lines with potential application code (= function bodies), and then 
   strip out all comments. By doing that, we get a reasonable approximation of actual lines of code.
   
   This module does the code detection step."
  (:require [evolutionary-metrics.complexity.loco :as loco]
            [clojure.string :as s]))


(def ^:private constantly-false (constantly false))

(defn- has-pending?
  [{:keys [pending] :as _acc}]
  pending)

(defn- pend-for
  [acc]
  (assoc acc :pending true))

(defn- reset-pending
  [acc]
  (assoc acc :pending false))

(defn- terminates-multi-comment?
  [multi-end?
   acc
   text]
  (and (has-pending? acc)
       (multi-end? text)))

(defn- multi-style-on-one-line?
  [multi-start? multi-end? text]
  (and (multi-start? text) ; e.g. /* a one liner multi line style */
       (multi-end? text)))

(defn- ignore-comment
  [acc _a-line]
  acc)

(defn- keep-code-in
  [{:keys [code] :as acc}
   a-line]
  (assoc acc :code (into code [a-line])))

(defn- non-comment-lines-in
  "Quite tricky since we need to support multi-line comments too"
  [file-content
   {:keys [comment? multi-start? multi-end?]
    :or   {multi-start? constantly-false multi-end? constantly-false} :as _language-rules}]
  (let [only-code-lines
        (->> file-content
             s/split-lines
             (map-indexed (fn [index the-line]
                            {:line-number (inc index)
                             :line the-line}))
             (remove (comp s/blank? :line))
             (reduce (fn [acc {:keys [line] :as a-line}]
                       (cond
                         (terminates-multi-comment? multi-end? acc line)
                         (reset-pending acc)

                         (has-pending? acc)
                         (ignore-comment acc a-line)

                         (comment? line)
                         (ignore-comment acc a-line)

                         (multi-style-on-one-line? multi-start? multi-end? line)
                         (ignore-comment acc a-line)

                         (multi-start? line)
                         (pend-for acc)

                         :else (keep-code-in acc a-line)))
                     {:code    []
                      :pending false}))
        the-code (:code only-code-lines)
        lines-with-code (->> the-code (map :line-number))]
    lines-with-code))

(defn find-non-commented-lines-in
  "Returns a set which contains the line numbers of each -- potentially -- executable 
   line in the given file. (We still don't know much about declarations...)."
  [file-path file-content]
  (let [language-rules (loco/language-rules-for file-path)]
    (non-comment-lines-in file-content language-rules)))