(ns codescene.features.client.oauth2-routes
  "Routes for OAuth2 flows and their implementations."
  (:require [clj-http.client :as http]
            [clojure.core.memoize :as memo]
            [codescene.features.util.maps :refer [map-of]]
            [codescene.features.components.kv-store :as kv-store]
            [codescene.features.util.url :as url-utils]
            [codescene.features.util.http-helpers :as http-helpers]
            [codescene.features.client.oauth2 :as oauth2]
            [codescene.features.components.http :as http-component]
            [compojure.core :refer [context make-route]]
            [ring.util.response :as response]
            [taoensso.timbre :as log]))

;; RFC 6749 based OAuth2

;;; request handlers

;; 1. Redirect user to request GitHub access:
;;    GET https://github.com/login/oauth/authorize?state=<random string>&redirect_uri=http://localhost:4000/repo-providers/github/callback&...
;; 2. GitHub redirects back to your site:
;;    GET http://localhost:4000/repo-providers/github/callback?code=<temp code>&state=<random string>
;; 3. Exchange this for an access token:
;;    POST https://github.com/login/oauth/access_token?code=<temp code>&state=<random string>&redirect_uri=http://localhost:4000/repo-providers/github/callback&...
;; 4. We redirect user to new project page:
;;    GET http://localhost:4000/select-repos

;; OAuth2 sites like bitbucket work the same

(defn authorize-redirect
  "This should be mounted as a route to start the OAuth2 flow. The user will be redirected to
  provider. This function generates that redirect:
  - generate state, add it to redirect and kv-store
  - take 'next' url from the incoming request and store it into session to redirect user to appropriate page after flow is done
  - params are passed to oauth-provider object to use in scope calculation

  OAuth2 flow has two nonces. One is state that is passed from client to AS, then AS passes it back. It is used to
  verify that a response from AS is one we have initiated. The other is the PKCE code verifier. We submit it in the
  initial request to AS and then with the final request to AS. We need to pair it with state to figure out which
  authentication attempt code challenge belongs to."
  [{:keys [flash params session] :as req} oauth2-provider kv-store]
  (-> (if (:session/key req)
        (let [{:keys [state code-verifier] :as auth-flow} (oauth2/init-auth-flow req)]
          (kv-store/set-val kv-store state auth-flow)
          (log/infof "Authorizing %s access with params %s" oauth2-provider (pr-str params))
          (doto (oauth2/provider-authorize-url oauth2-provider params state code-verifier)
            log/trace))
        ;; if no session redirect to itself and it'll have a session
        (do (log/info "No session, redirecting to itself")
            (url-utils/request-url req)))
      (response/redirect)
      (response/header "Cache-Control" "no-store")
      (assoc :session (or session {}))
      (assoc :flash flash)))

(def use-authorization-code
  "Concurrent login flows for Azure will sometimes get same authorization code concurrently
   and when you go to redeem this code multiple times, some requests get an error,
   that access token was already issued.

   To prevent this as much as possible we can use memoization, but this falls short when concurrent
   requests go to different JVMs. Use something like memento redis for that.

   This mostly happens when automated integration tests try to login with same user concurrently."
  (memo/ttl
    (with-meta
      (fn [oauth2-provider http-client-opts code {:keys [state code-verifier]}]
        (log/debugf "Requesting %s access token using code %s" (oauth2/oauth2-provider-id oauth2-provider) code)
        (let [client-params (merge http-client-opts
                                   (oauth2/authorization-code-req oauth2-provider code state code-verifier)
                                   (when-let [timeout-ms (oauth2/oauth2-provider-timeout oauth2-provider)]
                                     {:socket-timeout timeout-ms
                                      :connection-timeout timeout-ms})
                                   {:throw-entire-message? true})
              _ (log/trace client-params)
              response (http-helpers/with-http-error-messages "Failed to request OAuth2 token."
                         (http/request client-params))]
          (when-not (= (:status response) 200)
            (throw (ex-info (format "Failed to request OAuth2 token for provider %s, invalid http response."
                                    (oauth2/oauth2-provider-id oauth2-provider))
                            {:response response})))
          (:body response)))
      {::memo/args-fn #(nth % 2)})
    :ttl/threshold 10000))

(defn handle-callback
  "Finishes the OAuth2 login flow. If it finishes successfully, finish-callback will
  be called with request, OAuth2Provider, a map with token info and next URL. The result
  should be a ring response. Kv-store is codescene.features.components.kv-store/KVStore instance."
  [{:keys [system params] :as req}
   oauth2-provider
   kv-store
   finish-callback
   error-callback]
  (log/debugf "Handle OAuth2 callback for %s" (oauth2/oauth2-provider-id oauth2-provider))
  (try
    (let [provider-name (oauth2/oauth2-provider-id oauth2-provider)
          _ (oauth2/assert-callback-params! params provider-name)
          {:keys [code state]} params
          flow (kv-store/remove-val kv-store state)
          {:keys [code-verifier next-url]} (oauth2/assert-valid-flow! flow (:session/key req) provider-name)
          {:keys [access_token scope scopes refresh_token id_token
                  error error_description] :as access-token-response}
          (use-authorization-code oauth2-provider (http-component/client-config system) code (map-of state code-verifier))]
      ;; http://tools.ietf.org/html/draft-ietf-oauth-v2-31#section-5.1
      ;; https://openid.net/specs/openid-connect-core-1_0.html#TokenResponse
      ;; azure has access_token, refresh_token, scope, token_type ("jwt-bearer"), expires_in (3599)
      ;; github has access_token scope
      ;; bitbucket has access_token, expires_in (7200), refresh_token, scopes
      ;; gitlab has access_token token_type refresh_token scope created_at id_token
      ;; id_token is OpenID connect token, appears if openid scope was requested
      (if access_token
        (finish-callback req
                         oauth2-provider
                         {:access-token access_token
                          :refresh-token refresh_token
                          :id-token id_token
                          :scope (or scope scopes)}
                         next-url)
        ;; if the authorization code is invalid you'll get something like this:
        ;; {:error "bad_verification_code", :error_description "The code passed is incorrect or expired.",
        ;;  :error_uri "https://developer.github.com/apps/managing-oauth-apps/troubleshooting-oauth-app-access-token-request-errors/#bad-verification-code"}
        (throw (ex-info (format "Failed to request OAuth2 token. (%s, %s)" error error_description)
                        {:response access-token-response
                         :code (if (= "bad_verification_code" error) code "XXX")}))))
    (catch Exception e
      (log/warn e)
      (error-callback req oauth2-provider e))))

(defn routes
  "Generates routes for OAuth2 given
   - oauth2-provider
   - a finishing function that receives:
         - the request
         - oauth2-provider
         - db-style token definition
         - next URI
     and should return a response.
   - an error function that receives:
          - the request
          - oauth2-provider
          - exception that was caught
     and should return a response.

   Expects request to contain key :http-client with map of http client settings."
  [oauth2-provider kv-store finish-callback error-callback]
  ;; the reason to use make-route function because macros get evaluated at compile time and
  ;; they fail because the oauth2-mount-point is not implemented on keyword at that time
  (context "/" []
    (make-route
      :get
      (str (second (oauth2/oauth2-mount-point oauth2-provider)) "/authorize")
      (fn [req] (authorize-redirect req oauth2-provider kv-store)))
    (make-route
      :get
      (str (second (oauth2/oauth2-mount-point oauth2-provider)) "/callback")
      (fn [req] (handle-callback req oauth2-provider kv-store finish-callback error-callback)))))
