(ns codescene.features.chat.service.mcp.protocol
  "Generic MCP JSON-RPC protocol implementation."
  (:require [clj-http.client :as http]
            [clojure.data.json :as json]
            [clojure.string :as str]
            [taoensso.timbre :as log])
  (:import (java.util UUID)))

(defn- build-headers [{:keys [session-id headers]}]
  (cond-> {"Content-Type" "application/json"
           "Accept" "application/json, text/event-stream"}
    session-id (assoc "mcp-session-id" session-id)
    headers (merge headers)))

(defn- parse-sse-data [body-str]
  (let [lines (str/split-lines body-str)
        data-lines (filter #(str/starts-with? % "data:") lines)]
    (when (> (count data-lines) 1)
      (log/warnf "SSE has %d data lines but only using first!" (count data-lines)))
    (when (seq data-lines)
      (-> (first data-lines)
          (subs 5)
          str/trim
          (json/read-str :key-fn keyword)))))

(defn- parse-json-response [body-str content-type]
  (let [json-data (if (str/includes? content-type "text/event-stream")
                    (parse-sse-data body-str)
                    (json/read-str body-str :key-fn keyword))]
    (if (:error json-data)
      {:success false :error (get-in json-data [:error :message] "MCP error")}
      {:success true :result (:result json-data)})))

(defn- handle-success-response [response]
  (parse-json-response (:body response)
                       (get-in response [:headers "content-type"] "")))

(defn- invalid-session-error?
  "Detects session-related errors that require re-initialization.
   These occur when:
   - The MCP server has restarted and the old session is stale
   - The session ID is missing or invalid
   - A request is made before initialization completes"
  [status body]
  (and (#{400 404} status)
       (string? body)
       (or (str/includes? body "No valid session ID")
           (str/includes? body "Missing session ID")
           (str/includes? body "before initialization")
           (str/includes? body "Invalid session")
           (str/includes? body "session not found")
           (str/includes? body "Unknown session"))))

(defn- handle-response [response on-new-session]
  (let [status (:status response)
        headers (:headers response)
        body (:body response)
        ;; clj-http lowercases headers
        response-session (or (get headers "mcp-session-id")
                             (get headers "Mcp-Session-Id"))]
    (when (and response-session on-new-session)
      (on-new-session response-session))
    (case status
      200 (-> (handle-success-response response)
              (assoc :session-id response-session))
      202 {:success true :result {:status "accepted"} :session-id response-session}
      (cond-> {:success false :error (format "MCP status %d: %s" status body)}
        (invalid-session-error? status body) (assoc :invalid-session true)))))

(defn- build-request-body [method params]
  {:jsonrpc "2.0"
   :id (str (UUID/randomUUID))
   :method method
   :params (or params {})})

(defn- send-request*
  "Internal function that performs the actual HTTP request."
  [{:keys [url session-id headers on-new-session http-opts]} method params]
  (let [all-headers (build-headers {:session-id session-id :headers headers})
        body (build-request-body method params)
        request-opts (merge {:body (json/write-str body)
                             :headers all-headers
                             :socket-timeout 30000
                             :connection-timeout 5000
                             :throw-exceptions false}
                            http-opts)]
    (log/debugf "MCP request: method=%s url=%s params=%s" method url (pr-str params))
    (try
      (let [response (http/post url request-opts)
            result (handle-response response on-new-session)]
        (log/debugf "MCP response: method=%s result=%s" method (pr-str result))
        result)
      (catch Exception e
        (log/warn e "MCP request failed")
        {:success false :error (str "MCP connection failed: " (.getMessage e))}))))

(defn send-request
  "Sends a JSON-RPC request to an MCP server (without retry logic).
   req-ctx keys: :url, :session-id, :headers, :on-new-session, :http-opts"
  [req-ctx method params]
  (send-request* req-ctx method params))

(defn- send-initialized-notification [{:keys [url session-id headers http-opts]}]
  (let [all-headers (build-headers {:session-id session-id :headers headers})
        request-opts (merge {:body (json/write-str {:jsonrpc "2.0" :method "notifications/initialized"})
                             :headers all-headers
                             :socket-timeout 5000
                             :connection-timeout 5000
                             :throw-exceptions false}
                            http-opts)]
    (try
      (http/post url request-opts)
      (catch Exception e
        (log/warn e "Failed to send initialized notification")))))

(defn- perform-initialization
  "Performs the MCP initialization handshake."
  [req-ctx]
  (log/info "Initializing MCP session...")
  (let [init-result (send-request req-ctx
                                  "initialize"
                                  {:protocolVersion "2024-11-05"
                                   :capabilities {:tools {}}
                                   :clientInfo {:name "CodeScene Chat" :version "1.0.0"}})]
    (when-not (:success init-result)
      (log/warnf "Failed to initialize MCP: %s" (:error init-result)))
    (when-let [new-session-id (:session-id init-result)]
      (log/info "MCP session initialized")
      (send-initialized-notification (assoc req-ctx :session-id new-session-id))
      new-session-id)))

(defn ensure-initialized
  "Ensures MCP session is initialized.
   Returns the session-id (existing or newly created).
   req-ctx keys: :url, :session-id, :headers, :on-new-session"
  [{:keys [session-id] :as req-ctx}]
  (or session-id (perform-initialization req-ctx)))

(defn send-request-with-retry
  "Sends a JSON-RPC request to an MCP server with automatic session recovery.
   If the session is invalid, resets it, reinitializes, and retries once.
   req-ctx keys: :url, :session-id, :headers, :on-new-session, :http-opts, :on-reset-session"
  [{:keys [on-reset-session] :as req-ctx} method params]
  (let [result (send-request req-ctx method params)]
    (if (:invalid-session result)
      (do
        (log/info "MCP session invalid, resetting and reinitializing...")
        (when on-reset-session (on-reset-session))
        ;; Remove the stale session, reinitialize, then retry with new session
        (let [fresh-ctx (dissoc req-ctx :session-id)
              new-session-id (ensure-initialized fresh-ctx)]
          (send-request (assoc fresh-ctx :session-id new-session-id) method params)))
      result)))
