(ns codescene.features.chat.service.llm.gemini
  "Google Gemini API provider."
  (:require [clojure.data.json :as json]
            [clojure.string :as str]
            [codescene.features.chat.service.llm.http :as http]
            [codescene.features.chat.service.llm.provider :as provider]
            [taoensso.timbre :as log]))

(defn- role->gemini-role
  "Converts internal role to Gemini API role.
   Gemini uses 'user' and 'model' (not 'assistant')."
  [role]
  (case role
    :user "user"
    :assistant "model"
    :system "user" ; system messages handled separately
    "user"))

(defn- format-content
  "Formats a single message for Gemini's contents format."
  [{:keys [role content]}]
  {:role (role->gemini-role role)
   :parts [{:text content}]})

(defn- extract-system-instruction
  "Extracts the system message for Gemini's systemInstruction field."
  [messages]
  (when-let [system-msg (first (filter #(= :system (:role %)) messages))]
    {:parts [{:text (:content system-msg)}]}))

(defn- remove-system-messages
  "Removes system messages - they go in systemInstruction."
  [messages]
  (remove #(= :system (:role %)) messages))

(defn- parse-response
  "Parses a Gemini API response into our standard format.
   Extracts content and usage metadata."
  [response]
  (if (= 200 (:status response))
    {:success true 
     :content (get-in response [:body :candidates 0 :content :parts 0 :text])
     :usage {:prompt-tokens (get-in response [:body :usageMetadata :promptTokenCount])
             :completion-tokens (get-in response [:body :usageMetadata :candidatesTokenCount])}}
    {:success false
     :error (or (get-in response [:body :error :message])
                (str "API error: " (:status response) " - " (:body response)))}))

(defn- build-request-body
  "Builds the Gemini API request body with proper contents and systemInstruction."
  [messages]
  (let [system-instruction (extract-system-instruction messages)
        conversation-msgs (remove-system-messages messages)
        contents (mapv format-content conversation-msgs)]
    (cond-> {:contents contents}
      system-instruction (assoc :systemInstruction system-instruction))))

(defn- send-request
  "Sends a request to the Gemini API."
  [url request-body http-opts]
  (http/post url
             (merge {:headers {"Content-Type" "application/json"}
                     :body (json/write-str request-body)
                     :as :json}
                    http-opts)))

(defrecord GeminiProvider []
  provider/LLMProvider
  
  (call [_this {:keys [base-url chat-endpoint api-key model http-opts]} messages]
    (let [url (str base-url (str/replace chat-endpoint "{model}" model) "?key=" api-key)
          request-body (build-request-body messages)]
      (log/debugf "Calling Gemini API")
      (let [response (send-request url request-body http-opts)]
        (if (http/request-failed? response)
          (http/format-request-error response)
          (parse-response response))))))
