(ns codescene.features.api.routes.chat
  "Chat REST API routes for the AI assistant.
   Uses Bearer token (PAT) authentication like other REST API routes.
   The frontend passes the auth-token which is used for MCP server calls."
  (:require [codescene.features.api.core :as api-core]
            [codescene.features.api.privileges :as api-privileges]
            [codescene.features.api.spec.chat :as chat-spec]
            [codescene.features.chat.service :as chat-service]
            [codescene.features.components.chat :as chat]
            [codescene.features.util.api :as api-utils]
            [compojure.api.sweet :refer [context GET POST]]
            [spec-tools.data-spec :as ds]
            [taoensso.timbre :as log]))

(defn- get-chat-component [system]
  (chat/component system))

(defn- normalize-role [role]
  (if (string? role)
    (keyword role)
    role))

(defn- parse-message [message]
  {:role    (normalize-role (or (:role message) (get message "role")))
   :content (or (:content message) (get message "content"))})

(defn- parse-messages [messages]
  (mapv parse-message messages))

(defn- get-config-value
  "Gets a value from a config map, checking both keyword and string keys."
  ([config k] (get-config-value config k nil))
  ([config k default]
   (or (get config k) (get config (name k)) default)))

(defn- parse-llm-config [llm-config]
  (when llm-config
    (let [provider (get-config-value llm-config :provider "openai")
          model (get-config-value llm-config :model)
          api-key (get-config-value llm-config :api_key)
          base-url (get-config-value llm-config :base_url)]
      (cond-> {:provider (keyword provider)
               :model model}
        api-key (assoc :api-key api-key)
        base-url (assoc :base-url base-url)))))

(defn- send-chat-message
  "Handle chat message request. The auth_token for MCP is passed in the request body.
   opts is an optional map that can contain :mcp-config to pass server-specific configuration.
   mcp-config format: {:server-name {:headers {...}}}"
  [system request opts]
  (let [body (:body-params request)
        options (cond-> {}
                  (:llm_config body) (assoc :llm-config (parse-llm-config (:llm_config body)))
                  (:project_id body) (assoc :project-id (:project_id body))
                  (:auth_token body) (assoc :auth-token (:auth_token body))
                  (:instance_url body) (assoc :instance-url (:instance_url body))
                  (:mcp-config opts) (assoc :mcp-config (:mcp-config opts)))]
    (log/debugf "Chat request: conversation=%s, messages=%d" (:conversation_id body) (count (:messages body)))
    (-> (chat/send-message (get-chat-component system)
                           (:conversation_id body)
                           (parse-messages (:messages body []))
                           options)
        api-utils/ok)))

(defn- list-mcp-prompts
  "Handle MCP prompts list request."
  [request opts]
  (let [query-params (:query-params request)
        prompts (chat-service/list-prompts
                 {:auth-token (:auth_token query-params)
                  :instance-url (or (:instance_url query-params)
                                    (get-in request [:headers "x-cs-instance-url"]))
                  :mcp-config (:mcp-config opts)})]
    (log/debugf "MCP prompts request: returned %d prompts" (count prompts))
    (api-utils/ok {:prompts prompts})))

(defn- get-mcp-prompt
  "Handle MCP prompt get request."
  [request prompt-name opts]
  (let [query-params (:query-params request)
        config {:prompt-name prompt-name
                :auth-token (:auth_token query-params)
                :instance-url (or (:instance_url query-params)
                                  (get-in request [:headers "x-cs-instance-url"]))}
        result (chat-service/get-prompt (assoc config :mcp-config (:mcp-config opts)))]
    (log/debugf "MCP prompt get request: prompt=%s" prompt-name)
    (if (:success result)
      (api-utils/ok result)
      (api-utils/bad-request (:error result)))))

(defn sub-routes
  "Chat routes at /chat level.
   opts is an optional map that can contain :mcp-config for MCP server configuration."
  ([system] (sub-routes system {}))
  ([system opts]
   (context "/chat" []
     :tags ["chat"]
     :middleware [#(api-core/wrap-authorize-any system % #{api-privileges/restapi-read
                                                           api-privileges/mcp-api-read})]
     (POST "/" request
       :summary "Send a chat message"
       :description "Send a message to the AI assistant with optional tool use"
       :body [data (ds/spec {:spec ::chat-spec/chat-request :name :chat-request})]
       :responses {200 {:schema ::chat-spec/chat-response
                        :description "Chat response with assistant message and optional tool calls"}}
       (send-chat-message system request opts))

     (GET "/prompts" request
       :summary "List available MCP prompts"
       :description "Lists all prompts available from MCP servers"
       :query-params [auth_token :- ::chat-spec/auth_token
                      {instance_url :- ::chat-spec/instance_url nil}]
       :responses {200 {:schema ::chat-spec/prompts-list-response
                        :description "List of available prompts from MCP servers"}}
       (list-mcp-prompts request opts))

     (GET "/prompts/:prompt-name" request
       :summary "Get a specific MCP prompt"
       :description "Gets a prompt by name, returning the messages ready to use"
       :path-params [prompt-name :- ::chat-spec/prompt_name]
       :query-params [auth_token :- ::chat-spec/auth_token
                      {instance_url :- ::chat-spec/instance_url nil}]
       :responses {200 {:schema ::chat-spec/get-prompt-response
                        :description "Prompt result with messages ready to use"}}
       (get-mcp-prompt request prompt-name opts)))))
