-- name: delete-developer!
delete from developers where id = ?;

-- name: delete-team!
delete from teams where id = ?;

-- name: insert-developer<!
-- Inserts the given developer.
insert into developers
       (name,  team, excluded, ex_developer, settings_id)
       values(:name, :team, :excluded, :ex_developer, :settings_id);

-- name: developers-in-settings
-- Retrieves all developers associated with given settings.
select d.id, d.name, d.excluded, d.ex_developer, t.name as team, t.id as teamid from developers as d
       left join teams as t
            on d.team = t.id
       where d.settings_id = ?;

-- name: all-developers
-- Returns a list of all developers defined for all settings.
select d.id, d.name, d.excluded, d.ex_developer, t.name as team, t.id as teamid from developers as d
       left join teams as t
            on d.team = t.id
       order by d.name asc;

-- name: developer-by-settings-and-alias-name
-- Returns a developer, if any, matching the given criteria.
-- "name" is matched against the developer's aliases
select d.id as developer_id, d.name, d.excluded, d.ex_developer,
       t.name as team,
       t.id as teamid
       from developers as d
	join developer_aliases as da
	     on d.id = da.developer
       left join teams as t
            on d.team = t.id
       where d.settings_id = :settings_id and da.alias = :developer_name;

-- name: developer-by-settings-and-dev-name
-- Returns a developer, if any, matching the given criteria.
-- "name" is matched against the developer's aliases
select d.id as developer_id, d.name, d.excluded, d.ex_developer, 
       t.name as team,
       t.id as teamid
       from developers as d
	join developer_aliases as da
	     on d.id = da.developer
       left join teams as t
            on d.team = t.id
       where d.settings_id = :settings_id and d.name = :developer_name;

-- name: developers-by-settings-and-team-id
-- Returns all developers that are assigned to the given team.
select d.id, d.name, d.excluded, d.ex_developer,
       t.name as team,
       t.id as teamid 
       from developers as d
       left join teams as t
            on d.team = t.id
       where d.settings_id = ? and t.id = ?;

-- name: ex-developers-by-settings
-- Returns all previous developers who no longer work on a given project.
select d.id, d.name, d.excluded, d.ex_developer
       from developers d
       where d.settings_id = ?
              and d.ex_developer = 1;

-- name: mark-as-ex-developer!
update developers
       set ex_developer = 1
       where id = :developer_id;

-- name: update-developer!
-- Updates the team of the developer identified by the id.
update developers
       set team = :team, excluded = :excluded, ex_developer = :ex_developer
       where id = ?;

-- name: insert-team<!
-- Inserts the given team while returning its generated ID.
insert into teams
       (name, settings_id)
       values(:name, :settings_id);

-- name: teams-in-settings
-- Returns all teams defined for given settings
select t.id, t.name
       from teams as t
       where t.settings_id = ?
       order by name asc;

-- name: team-by-settings-and-name
-- Returns the team identified by its name in given settings.
select t.id, t.name
       from teams as t
       where t.settings_id = ? and t.name = ?;

-- name: update-team!
-- Updates the name of the team identified by the id.
update teams
       set name = :name
       where id = :id;

-- name: developer-aliases-by-settings
-- Returns each alias, joined to its corresponding developer.
select d.id, d.name, d.excluded, d.ex_developer, t.name as team, t.id as teamid, da.id as alias_id, da.alias
       from developers as d
       	    join developer_aliases da on d.id = da.developer
            left join teams as t on d.team = t.id
       where d.settings_id = :settings_id
       order by d.name, da.alias,  d.id asc;

-- name: developer-alias-by-settings-and-name
select da.alias, da.id, da.developer
       from developer_aliases da
       where da.alias = :alias and da.settings_id = :settings_id;

-- name: developer-alias-by-settings-and-id
select alias, id, developer
       from developer_aliases
       where id = :alias_id and settings_id = :settings_id;

-- name: update-alias!
update developer_aliases
	set alias = :alias, developer= :developer, settings_id = :settings_id
	    where id = :alias_id;
	    
-- name: aliases-by-developer-and-settings
select d.id, d.name, d.excluded, d.ex_developer, t.name as team, t.id as teamid, da.id as alias_id, da.alias
       from developers as d
       	    join developer_aliases da on d.id = da.developer
            left join teams as t on d.team = t.id
       where d.settings_id = :settings_id
       	     and
	     d.id = :developer_id
       order by d.name, da.alias,  d.id asc;


-- name: set-developer-alias!
update developer_aliases set developer = :developer_id
       where id = :alias_id and settings_id = :settings_id;

-- name: insert-developer-alias!
insert into developer_aliases (alias, settings_id, developer)
       values (:alias, :settings_id, :developer_id);

-- name: delete-missing-developers-by-settings!
-- Delete developers from a settings when they are not in the
-- list of known devs.
delete from developers 
       where settings_id = :settings_id
       	     and id not in (:developer_ids);

-- name: delete-missing-ex-developers-by-settings!
-- Delete ex-developers from settings when they are not in the
-- list of known ex-devs.
update developers set ex_developer = 0 
       where settings_id = :settings_id
              and id not in (:developer_ids);

-- name: rename-developer-by-id-settings!
update developers set name = :new_name
       where settings_id = :settings_id
       	     and id = :id;

-- name: delete-all-developer-settings!
delete from developer_settings;

-- name: delete-developer-settings!
delete from developer_settings where id = ?;

-- name: all-developer-settings
-- Retrieves all developer-settings.
select * from developer_settings;

-- name: update-last-project-id!
update developer_settings set last_project_id = :last_project_id
       where id = :settings_id

-- name: insert-project-developer-settings<!
-- Associates the given project with given settings
insert into project_developer_settings
       (settings_id, project_id)
       values(:settings_id, :project_id);
 
-- name: delete-project-developer-settings!
delete from project_developer_settings 
       where project_id = :project_id;

-- name: developer-settings-by-id
-- Retrieves specific developer-settings
select * from developer_settings where id = ?;

-- name: developer-settings-by-project
-- Retrieves developer-settings associated with a given project.
select ds.* from developer_settings as ds
       left join project_developer_settings as pds
              on ds.id = pds.settings_id
       where pds.project_id = :project_id;

-- name: projects-by-developer-settings
-- Retrieves projects associated with given developer-settings.
select p.id as project_id,
       p.name as project_name
  from project_developer_settings pds join projects p on p.id = pds.project_id
 where pds.settings_id = ?;

-- name: insert-developer-settings<!
-- Inserts the given developer settings.
insert into developer_settings
       (name)
       values(:name);

-- name: update-developer-settings!
update developer_settings set name = :name, description = :description
       where id = :id;
