-- name: select-all-providers
-- Returns basic data for all authentication providers.
-- More details can be fetched with `select-provider-config` and `select-provider-role-mapping`.
select p.id as provider_id, p.type as provider_type, p.active as active, p.priority_order,
  r.id as role_id, r.name as role_name
from auth_providers p
  left join roles r on r.id = p.default_role_id
order by p.priority_order;

-- name: select-provider-config
-- Selects configuration properties for given authentication provider
select pc.`key`, pc.`value`
from auth_provider_config pc
  join auth_providers p on pc.provider_id = p.id
where p.id = :provider_id;

-- name: select-provider-role-mapping
-- Fetches mapping between provider's specific roles to CodeScene internal roles
-- `role_id` and `role_name` represents CodeScene's authorization roles while `provider_role_name`
--  represents external provider "role name" (e.g. name of corresponding Active Directory group
--  used for CodeScene users)
select r.id as codescene_role_id, r.name as codescene_role_name, prm.group_id as group_id, prm.project_id as project_id, prm.provider_role_name as provider_role_name
from auth_provider_role_mapping prm
  join auth_providers p on prm.provider_id = p.id
  join roles r on r.id = prm.codescene_role_id
where p.id = :provider_id;

-- name: save-provider<!
-- Inserts new provider
INSERT INTO auth_providers (type, active, default_role_id)
VALUES (:provider_type, :provider_active, :provider_default_role_id);

-- name: update-provider!
-- update an existing provider
UPDATE auth_providers
SET active = :provider_active, default_role_id = :provider_default_role_id
WHERE type = :provider_type;

-- name: update-provider-default-role!
-- update default role for existing provider
UPDATE auth_providers
SET default_role_id = :default_role_id
WHERE id = :id;

-- name: insert-provider-config-property!
-- Inserts a single key-value pair representing specifig config property for given provider
INSERT INTO auth_provider_config (provider_id, `key`, `value`)
VALUES (:provider_id, :provider_key, :provider_value);

-- name: update-provider-config-property!
-- Updates a single key-value pair representing specifig config property for given provider
UPDATE auth_provider_config set `value` = :provider_value
WHERE provider_id = :provider_id AND `key` = :provider_key;

-- name: delete-provider-role-mapping!
-- Inserts new provider role-mapping
DELETE FROM auth_provider_role_mapping
WHERE provider_id = :provider_id AND group_id IS NULL AND project_id is null;

-- name: delete-provider-role-name-mapping!
-- Delet provider role-mapping for specific provider/name/group
DELETE FROM auth_provider_role_mapping
WHERE provider_id = :provider_id AND provider_role_name = :provider_role_name AND group_id IS NULL AND project_id IS NULL;

-- name: delete-all-provider-role-name-mappings!
-- Delet provider role-mappings for specific provider/name
DELETE FROM auth_provider_role_mapping
WHERE provider_id = :provider_id AND provider_role_name = :provider_role_name;

-- name: delete-group-role-name-mapping!
-- Delete provider role-mapping for specific provider/name and group
DELETE FROM auth_provider_role_mapping
WHERE provider_id = :provider_id AND provider_role_name = :provider_role_name AND group_id = :group_id;

-- name: delete-group-role-name-and-id-mapping!
-- Delete provider role-mapping for specific provider/name/role-id and group
DELETE FROM auth_provider_role_mapping
WHERE provider_id = :provider_id AND provider_role_name = :provider_role_name AND group_id = :group_id AND codescene_role_id =:role_id;

-- name: delete-project-role-name-mapping!
-- Delete provider role-mapping for specific provider/name and project
DELETE FROM auth_provider_role_mapping
WHERE provider_id = :provider_id AND provider_role_name = :provider_role_name AND project_id = :project_id;

-- name: delete-project-role-name-and-id-mapping!
-- Delete provider role-mapping for specific provider/name/role-id and project
DELETE FROM auth_provider_role_mapping
WHERE provider_id = :provider_id AND provider_role_name = :provider_role_name AND project_id = :project_id AND codescene_role_id =:role_id;

-- name: insert-provider-role-mapping!
-- Inserts new provider role-mapping
INSERT INTO auth_provider_role_mapping (provider_id, codescene_role_id, provider_role_name, group_id, project_id)
VALUES (:provider_id, :codescene_role_id, :provider_role_name, :group_id, :project_id);

-- name: role-mappings-for-group
-- Fetches role mappings withing specified group, for all identities regardless of provider
select
	PROVIDER_ROLE_NAME username,
	CODESCENE_ROLE_ID role_id,
	PROVIDER_ID,
	ap.`TYPE` provider_name
from
	auth_provider_role_mapping aprm join auth_providers ap on
	aprm.PROVIDER_ID = ap.ID and ap.ACTIVE = 1
where
	group_id = :group_id

-- name: role-mappings-for-project
select
	PROVIDER_ROLE_NAME username,
	CODESCENE_ROLE_ID role_id,
	PROVIDER_ID,
	ap.`TYPE` provider_name
from
	auth_provider_role_mapping aprm join auth_providers ap on
	aprm.PROVIDER_ID = ap.ID and ap.ACTIVE = 1
where
	project_id = :project_id

-- name: select-default-roles
-- Returns the default roles
SELECT * FROM default_roles;

-- name: delete-group-default-role!
-- Delete default roles for specific group
DELETE FROM default_roles
WHERE group_id = :group_id;

-- name: delete-project-default-role!
-- Delete default roles for specific project
DELETE FROM default_roles
WHERE project_id = :project_id;

-- name: insert-default-role!
-- Inserts new default role
INSERT INTO default_roles (role_id, group_id, project_id)
VALUES (:role_id, :group_id, :project_id);
