grammar Tcl;
options {superClass=hotspots_x_ray.languages.InterruptibleParser;}
// Top level entry point for X-Ray
//
translationunit
:
	expression* EOF
;

expression: comment
          | procedure
          | method
          | constructor
          | destructor
          | class_declaration
          | global_block
          | anything;

comment: HASH ~NEWLINE*? NEWLINE;

anything : ~(LeftBrace | RightBrace);

class_declaration: OO_CLASS 'create' class_name LeftBrace expression*? RightBrace;
class_name: ID | SCOPED_NAME;

procedure: PROC proc_name proc_args proc_body;
method: METHOD proc_name method_args proc_body;
method_args: function_param | proc_args;
constructor: CONSTRUCTOR proc_args proc_body;
destructor: DESTRCUTOR proc_body;

proc_name: ID | SCOPED_NAME | ID SCOPED_NAME | METHOD_SCOPED_NAME;

proc_args: LeftBrace function_param*? RightBrace;

function_param: function_param_name | function_param_with_default_value;
function_param_name: ID;
function_param_with_default_value: LeftBrace function_param_name arg_default_value RightBrace;
arg_default_value: list_seq | conditional_expr | anything;

proc_body: LeftBrace proc_body_statement*? RightBrace;

proc_body_statement : block_statement
                    | anything;

block_statement : LeftBrace proc_body_statement*? RightBrace;
global_block: LeftBrace expression*? RightBrace;

// Entry point for cyclomatic complexity.
// We keep this separate from the nested complexity since we want
// to match -and -or operators too.
//
cyclomatic_complexity_scope: cyclomatic_complexity_exprs* EOF;

cyclomatic_complexity_exprs: comment
                            | cyclomatic_complexity
                            | cyclomatic_block_statement
                            | anything;

// Parsing cyclomatic complexity
//
cyclomatic_complexity: conditionals | loops | logical_operators;
conditionals: IF | ELSEIF | SWITCH;
loops: FOR | FOREACH | WHILE;
logical_operators: LOGICAL_AND | LOGICAL_OR;

cyclomatic_block_statement: LeftBrace cyclomatic_complexity_exprs*? RightBrace;

// Entry point for the nested complexity calculations:
//
complexity_scope : complexity_exprs* EOF;

complexity_exprs: complexity_inducing_expr
                | complexity_block_statement
                | anything;

complexity_inducing_expr: if_statement
                        | else_statement
                        | switch_statement
                        | for_loop
                        | foreach_loop
                        | while_loop;

if_statement: (IF | ELSEIF) conditional_expr THEN? multi_line_conditional_expression;
else_statement: ELSE multi_line_conditional_expression;

switch_statement: SWITCH ~(LeftBrace)*? multi_line_conditional_expression;

for_loop: FOR for_loop_spec multi_line_conditional_expression;
for_loop_spec:  conditional_expr conditional_expr conditional_expr; // e.g. for { set a 10}  {$a < 20} {incr a}

// foreach is really tricky to parse:
//
// foreach i {a b c} {j k} {d e f g} {
//       lappend x $i $j $k
// }
foreach_loop: FOREACH foreach_var_name foreach_expression_scope multi_line_conditional_expression;
foreach_var_name: foreach_expression_scope;
foreach_expression_scope: (list_seq | conditional_expr | foreach_list_anything)+;

foreach_list_anything:  ~(LeftBrace | RightBrace
                          | SET
                          | FOR | FOREACH | WHILE
                          | IF
                          | SWITCH
                          | PROC
                          | METHOD
                          | 'break' | 'continue' | 'default' | 'error');

list_seq: '[' foreach_expression_scope ']';

while_loop: WHILE conditional_expr multi_line_conditional_expression;

conditional_expr: LeftBrace conditional_expr_part*? RightBrace;
conditional_expr_part: logical_operators
                     | anything // foreach_list_anything
                     | conditional_expr;
multi_line_conditional_expression: LeftBrace complexity_exprs*? RightBrace;

complexity_block_statement : LeftBrace complexity_exprs*? RightBrace;

fragment ESCAPED : '\\\\' | '\\"';
STRING : '"' ( ESCAPED | ~('\n'|'\r') )*? '"';

Whitespace : [ \t]+ -> skip;

// TRICKY: prevent parsing regex as comment, example:  {^[ ]{0,3}#{1,6}}
//AlternateLineComment: '#' ~[\r\n{] ~[\r\n]* -> skip;
HASH: '#';

NEWLINE : '\r'? '\n';

PROC: 'proc';
SET: 'set';

FOR: 'for';
FOREACH: 'foreach';
IF: 'if';
THEN: 'then';
ELSEIF: 'elseif';
ELSE: 'else';
QUESTION_MARK: '?';
SWITCH: 'switch';
WHILE: 'while';
LOGICAL_AND: '&&';
LOGICAL_OR: '||';

NAMESPACE: 'namespace';
EVAL: 'eval';
OO_CLASS: '::oo::class';
CONSTRUCTOR: 'constructor';
DESTRCUTOR: 'destructor';
METHOD: 'method';

SPECIAL_ID: '%' [a-zA-Z_\-$][a-zA-Z0-9_\-$]*;

ID : [a-zA-Z_\-$][a-zA-Z0-9_\-$]*;

SCOPER : '::';
SCOPED_NAME : (SCOPER ID)+; // e.g. ::HelloWorld::MyProcedure

METHOD_SCOPER: '.';
METHOD_SCOPED_NAME: (ID METHOD_SCOPER)+ ID;

HEX_VALUE: '0' 'x' [0-9a-fA-F]+;

INTEGER: [0-9]+;

ESCAPED_LEFT_BRACE: '\\{';
ESCAPED_RIGHT_BRACE: '\\}';

LeftBrace: '{';
RightBrace: '}';

ANY_CHAR : .; // Put this lexer rule last to give it the lowest precedence
