grammar PowerShell;
options {superClass=hotspots_x_ray.languages.InterruptibleParser;}

// The function syntax for PowerShell is:
//
// function [<scope:>]<name> [([type]$parameter1[,[type]$parameter2])]
//   {
//     param([type]$parameter1 [,[type]$parameter2])
//     dynamicparam {<statement list>}
//     begin {<statement list>}
//     process {<statement list>}
//     end {<statement list>}
//   }


// Entry point for X-Ray and function based metrics:
//
translationunit
:
	expression* EOF
;

// Entry point for cyclomatic complexity.
// We keep this separate from the nested complexity since we want
// to match -and -or operators too.
//
cyclomatic_complexity_scope: cyclomatic_complexity_exprs* EOF;

cyclomatic_complexity_exprs: cyclomatic_complexity
                            | cyclomatic_block_statement
                            | anything;

// Entry point for the nested complexity calculations:
//
complexity_scope : complexity_exprs* EOF;

complexity_exprs: complexity_inducing_expr
                | complexity_block_statement
                | anything;

expression : function_declaration
           | named_script_block
           | test_code
           | global_data_declaration // used for LCOM4
           | global_block_statement
           | anything;

anything : ~(LeftBrace | RightBrace);

// Rules for parsing BDD style tests
test_code: test_fw_scope
          | unit_test
          | unit_test_fw_method;

test_fw_scope: CONTEXT context_name LeftBrace expression*? RightBrace;
context_name: LITERAL | SINGLE_LITERAL;

unit_test: IT unit_test_name function_body;
unit_test_name: LITERAL | SINGLE_LITERAL;

unit_test_fw_method: unit_test_fw_method_name function_body;
unit_test_fw_method_name: BEFORE_ALL | AFTER_ALL | BEFORE_EACH | AFTER_EACH;

// Rules for parsing plain functions
function_declaration: FUNCTION function_name function_args? function_body;
function_name: ID | SCOPED_NAME;

function_args: '(' function_definition_params_list? ')';

function_definition_params_list : function_param
                                | function_definition_params_list ',' function_param
                                ;

function_param: param_meta_data_types '$' function_param_name default_param_value?;
function_param_name: ID;

// A param can have a bunch of metadata specs:
// [Parameter(ParameterSetName="CellColor")]
// [ValidateNotNull()]
// [PoshGitCellColor]
param_meta_data_types: param_meta_data_type*;
param_meta_data_type: '[' ~']'*? ']'; // e.g. param ([switch]$on)

default_param_value: '=' ~','*?;

function_body: LeftBrace function_body_statement*? RightBrace;

function_body_params: PARAM '(' function_definition_params_list ')';

function_body_statement : function_body_params
                        | block_statement
                        | anything;

block_statement : LeftBrace function_body_statement*? RightBrace;
global_block_statement: LeftBrace expression*? RightBrace;

// A script block is a named element that we'd like to parse:
// PoshGitVcsPrompt = { ... }
named_script_block: '$' function_name '=' function_body;

// Capture global data declarations for LCOM4 calculations.
global_data_declaration: global_variable_name ('=' | '+=' | '-='); // assignment of event listeners
global_variable_name: plain_variable_name
                    | variable_name_with_reserved_char;
plain_variable_name: '$' variable_name;
variable_name_with_reserved_char: '$' LeftBrace variable_name RightBrace;
variable_name: SCOPED_NAME | ID;

// Parsing cyclomatic complexity
//
cyclomatic_complexity: conditionals | loops | logical_operators | exceptions;
conditionals: IF | ELSEIF | SWITCH;
loops: FOR | FOREACH | WHILE; // DO ... WHILE is handled implicitly by matching WHILE
logical_operators: LOGICAL_AND | LOGICAL_OR;
exceptions: CATCH;

cyclomatic_block_statement: LeftBrace cyclomatic_complexity_exprs*? RightBrace;

// Parsing nested complexity
//
complexity_inducing_expr: if_statement
                        | else_statement
                        | switch_statement
                        | for_loop
                        | do_while_loop
                        | while_loop;

if_statement: (IF | ELSEIF) conditional_expr multi_line_conditional_expression;
else_statement: ELSE multi_line_conditional_expression;

switch_statement: SWITCH conditional_expr multi_line_conditional_expression;

for_loop: (FOR | FOREACH) conditional_expr multi_line_conditional_expression;

do_while_loop: DO multi_line_conditional_expression WHILE conditional_expr_compounds?;
while_loop: WHILE conditional_expr multi_line_conditional_expression;

conditional_expr: conditional_expr_compounds
                | (conditional_operator
                  | ~LeftBrace)*?;
conditional_operator: LOGICAL_AND | LOGICAL_OR;
conditional_expr_compounds: '(' conditional_expr_compound ')';
conditional_expr_compound: conditional_expr_compounds
                         | (conditional_operator
                            | ~LeftBrace)*?;


multi_line_conditional_expression: LeftBrace complexity_exprs*? RightBrace;

complexity_block_statement : LeftBrace complexity_exprs*? RightBrace;

// Lexer
//

fragment ESCAPED_QUOTE : '`"';
LITERAL : '"' ( ESCAPED_QUOTE | ~('\n'|'\r') )*? '"';

fragment ESCAPED_SINGLE_LITERAL : '\'\'';
SINGLE_LITERAL : '\''  ( ESCAPED_SINGLE_LITERAL | ~('\n'|'\r') )*? '\'';

LeftBrace : '{';
RightBrace : '}';

Whitespace : [ \t]+ -> skip;

// Easier to capture this in the lexer. Used for metadata on parameters like [psobject[]]$Object
ARRAY_SPEC: '[' ']';

BlockComment: '<#' .*? '#>' -> skip;
LineComment: '#' ~[\r\n]* -> skip;

NEWLINE : '\r'? '\n' -> skip;

// PowerShell is case insensitive
fragment A : [aA];
fragment B : [bB];
fragment C : [cC];
fragment D : [dD];
fragment E : [eE];
fragment F : [fF];
fragment G : [gG];
fragment H : [hH];
fragment I : [iI];
fragment J : [jJ];
fragment K : [kK];
fragment L : [lL];
fragment M : [mM];
fragment N : [nN];
fragment O : [oO];
fragment P : [pP];
fragment Q : [qQ];
fragment R : [rR];
fragment S : [sS];
fragment T : [tT];
fragment U : [uU];
fragment V : [vV];
fragment W : [wW];
fragment X : [xX];
fragment Y : [yY];
fragment Z : [zZ];


FUNCTION: F U N C T I O N;
PARAM: P A R A M;

// Complexity inducing elements
IF: I F;
ELSE: E L S E;
ELSEIF: E L S E I F;
SWITCH: S W I T C H;
FOR: F O R;
FOREACH: F O R E A C H;
DO: D O;
WHILE: W H I L E;

CATCH: C A T C H;

LOGICAL_AND: '-' A N D;
LOGICAL_OR: '-' O R;

// Tokens for parsing test code
IT: I T;
CONTEXT: C O N T E X T;
BEFORE_EACH: B E F O R E E A C H;
AFTER_EACH: A F T E R E A C H;
BEFORE_ALL: B E F O R E A L L;
AFTER_ALL: A F T E R A L L;

ID : [a-zA-Z_][a-zA-Z0-9_\-?]*;
SCOPER : ':';
SCOPED_NAME : ID (SCOPER ID)+;
INTEGER: [0-9]+;

ANY_CHAR : .; // Put this lexer rule last to give it the lowest precedence
