(ns codescene.features.util.http-helpers
  "Helpers used "
  (:require [clojure.string :as string]
            [codescene.util.json :as json]
            [slingshot.slingshot :refer [throw+ try+]])
  (:import (java.net URL)))

(defn- without-bom
  "Remove the BOMs that azure for some reason includes in error responses"
  [s]
  (if (string/starts-with? s "\uFEFF")
    (subs s 1)
    s))

(defn extract-errors
  "Extract errors descriptions passed in http responses.
   Using a :message is entry is common, but all providers seem to have there own take on this."
  [headers body]
  (let [content-type (get headers "Content-Type" "")]
    (cond
      (string/includes? content-type "application/json")
      (try
        (let [body (json/parse-string (without-bom body))]
          (or
          ;; Jira on some errors
           (-> body :errorMessage)
          ;; Jira on other errors (lets extract just the first one for now)
           (-> body :errorMessages first)
          ;; GitLab on insufficient scope
           (-> body :error_description)
          ;; BitBucket on insufficient scope
           (-> body :error :message)
          ;; BitBucket on invalid URL
           (-> body :error :fields)
          ;; BitBucket Server on invalid URL (no PR)
           (->> body :errors (map :message) first)
          ;; GitLab on invalid URL
           (-> body :error)
          ;; ClickUp
           (-> body :err)
          ;; Everone uses this
           (-> body :message)))
       ;; Jira when unathenticated on some endpoints
       ;; Invalid json is return (it's just a string of text) 
        (catch Exception e
          body))

      (string/includes? content-type "text/plain")
      body

      (empty? content-type) ;; Assume no type means text...
      body)))

(defn with-http-error-messages*
  "Try to execute the body, catch http exceptions and rethrow with error message
   extracted from the http-response"
  [contextual-message body-fn]
  (try+
   (body-fn)
   (catch :status {:keys [status headers body] :as e}
     (let [errors (extract-errors headers body)
           msg (format "%s. Http status %d - %s"
                       contextual-message status
                       (if (not-empty errors)
                         errors "no details available"))]
       (throw+ {:type :http-error
                :message msg
                :status status
                :headers headers}
               msg)))
   ;; TODO: Should we be specific here, and only catch what can actually expected?
   ;; Note that this is not really a catch-all - throw+ lets ex-info:s not matched above pass this
   (catch Exception e
     (let [msg (format "%s. %s"  contextual-message (.getMessage e))]
       (throw+ {:type :http-error :message msg} msg)))))

(defmacro with-http-error-messages [contextual-message & body]
  `(with-http-error-messages* ~contextual-message
     (fn [] ~@body)))

(defn full-url-for [base-url extra-path]
  (let [url (URL. base-url)
        new-path (str (.getPath url) "/"  extra-path)
        full-url (URL. url (string/replace new-path #"//+" "/"))]
    (str full-url)))
