(ns codescene.features.util.file
  (:require [clojure.string :as str]
            [evolutionary-metrics.mining.file-patterns :as fp]))

(defn filter-by
  "filter filepaths using glob-patterns"
  [glob-patterns filepaths]
  (let [predicate (fp/path-glob-pred glob-patterns true)]
    (filter predicate filepaths)))

(defn exclude-by
  "exclude filenames using glob-patterns"
  [file-glob-patterns path-glob-patterns filenames]
  (let [file-predicate (fp/file-glob-pred file-glob-patterns false)
        path-predicate (fp/path-glob-pred path-glob-patterns false)]
    (filter #(-> (or (file-predicate %) (path-predicate %)) not) filenames)))

(defn exclude-by-supplier
  "exclude filenames using glob-patterns, the supplier should return the value to be matched against path-glob-patterns"
  [list supplier path-glob-patterns]
  (let [path-predicate (fp/path-glob-pred path-glob-patterns false)]
    (filter #(-> % supplier path-predicate not) list)))

(defn with-scope
  "add the last segment of the repo-path to filename, if repo-path is nil return filename unchanged"
  [repo-path filename]
  (cond->> filename
           repo-path (format "%s/%s" (fp/final-segment repo-path) filename)))

(defn file-names-with-scope
  "add the last segment of the repo-path to all filenames, if repo-path is nil return filenames unchanged"
  [repo-path filenames]
  (cond->> filenames
           ;;we skip processing entire filenames collection when there is no repo-path
           repo-path (map (partial with-scope repo-path))))

(defn file-names-de-scope
  "remove the scope from filenames"
  [filenames]
  (map #(-> (fp/de-scope %) second) filenames))

(defn is-absolute-path?
  [path]
  ;; Handles both / and C:/ style paths
  ;; (but assumes forward slashes)
  (some? (re-find #"(^/)|(^[A-Z]:/)" path)))

(defn is-relative-path?
  [path]
  (not (is-absolute-path? path)))

(defn is-simple-file-name
  "return true if path is just the file name without any folder structure"
  [path]
  (not (str/includes? path "/")))