(ns codescene.features.repository-provider.bitbucket-server.api
  "Wraps the bitbucket http API"
  (:require [clj-http.client :as http]
            [clj-time.core :as tc]
            [taoensso.timbre :as log]
            [codescene.features.client.api :as api-client :refer [to-url]]
            [codescene.features.util.http-helpers :as h]
            [evolutionary-metrics.trends.dates :as dates]
            [meta-merge.core :refer [meta-merge]]
            [org.clojars.roklenarcic.paginator :as page]
            [clojure.string :as str]))

(defn apply-client [authed-client params]
  (-> authed-client
      (api-client/augment-request params)
      (api-client/fix-api-url nil)))

(defn request-url [authed-client url]
  (:url (apply-client authed-client {:url url})))

(defn api-request*
  "Core function for API requests. Requires ApiClient token.

  Uses augment-request to add some sort of credentials to the request.
  If request returns 401 then on-expired-token will be called on auth-token.

  If that returns a ApiClient, the request will be retried."
  [authed-client method url request-params]
  (let [params (meta-merge
                 {:url url
                  :content-type :json
                  :accept       :json
                  :as           :json-strict
                  :request-method method}
                 request-params)
        final-params (apply-client authed-client params)]
    (when-not (:connection-manager final-params)
      (log/warnf "No connection manager, url=%s, avoid for production use." (:url final-params)))
    ;; we don't use renewing tokens here
    (h/with-http-error-messages (str "Failed to fetch Bitbucket data from " (:url final-params))
      (http/request final-params))))

(defn api-request
  [authed-client method url request-params]
  (:body (api-request* authed-client method url request-params)))

(defn api-page-request
  "A helper function for a request that requests a particular page. It is based on a paging state
  and returns an items+cursor map."
  [authed-client {:keys [cursor add-page]} url request-params]
  (log/debugf "Fetch page starting at item %s from %s" cursor url)
  (let [params (assoc-in request-params [:query-params :start] (or cursor 0))
        {:keys [isLastPage nextPageStart values]} (api-request authed-client :get url params)]
    (add-page values (when-not isLastPage nextPageStart))))

(defn- get-paged-data
  ([client url params]
   (get-paged-data client url params (constantly true)))
  ([client url params continue-fn]
   (loop [start 0
          previous-data []]
     (api-client/check-interrupt)
     (log/debugf "Fetch page starting at item %s from %s" start url)
     (let [{:keys [isLastPage nextPageStart values]} (api-request client :get url (assoc-in params [:query-params :start] start))
           all-data (into previous-data values)]
       (if (and (false? isLastPage) (continue-fn values))
         (recur nextPageStart all-data)
         all-data)))))

(defn- ->not-old-enough-fn [since]
  (fn [values]
    (when-let [updated-at (some-> values last :updatedDate dates/timestamp->date)]
      (tc/before? since updated-at))))

(defn get-all-pull-requests
  "Get all (outgoing) pull requests"
  [authed-client {:keys [owner-login repo-slug]} {:keys [since] :as _search-options}]
  (let [query-params {:direction "OUTGOING"
                      :state "ALL"} ;; TODO: get merged PRs only
        ;; Since there is no API for filtering pullrequests by update date
        ;; we have to manage it ourselves...
        continue-fn (if since
                      (->not-old-enough-fn since)
                      (constantly true))]
  (get-paged-data authed-client
                  (to-url "projects" owner-login "repos" repo-slug "pull-requests")
                  {:query-params query-params}
                  continue-fn)))

(defn get-open-pull-requests
  "Get open pull requests with specified source branch"
  [authed-client {:keys [owner-login repo-slug]} branch]
  (get-paged-data authed-client
                  (to-url "projects" owner-login "repos" repo-slug "pull-requests")
                  {:query-params
                   {:at branch
                    :direction "OUTGOING"}}))

(defn get-commit-ids
  "Returns a list of commit ids"
  [authed-client {:keys [owner-login repo-slug]} pr-id]
  (let [url (to-url "projects" owner-login "repos" repo-slug "pull-requests" pr-id "commits")]
    (vec (rseq (mapv :id (get-paged-data authed-client url {}))))))

(defn outdated-tag [run-tag] (str run-tag "# OUTDATED\n\n"))

(defn fix-existing-comment?
  "Comment cannot be deleted due to having replies. Let's mark it as outdated if it is not already.
   ignore exceptions"
  [status text run-tag]
  (and (= 409 status) (not (str/includes? text (outdated-tag run-tag)))))

(defn delete-comments-with-runtag
  "Deletes PR comments that contain the run-tag."
  [authed-client {:keys [owner-login repo-slug]} pr-id run-tag]
  (let [pr-url (to-url "projects" owner-login "repos" repo-slug "pull-requests" pr-id)
        comments (->> (get-paged-data authed-client (to-url pr-url "activities") {})
                      (filter #(= (:action %) "COMMENTED"))
                      (map :comment)
                      (group-by :id)
                      (reduce-kv #(conj %1 (apply max-key :version %3)) []))]
    (doseq [{:keys [id text version]} comments
            :when (str/includes? text run-tag)]
      (let [url (to-url pr-url "comments" id)
            {:keys [status]} (api-request* authed-client :delete url
                                           {:query-params {:version version}
                                            :unexceptional-status #(or (<= 200 % 399) (= 409 %))})]
        (when (fix-existing-comment? status text run-tag)
          (api-request authed-client :put url {:form-params {:text (str/replace text run-tag (outdated-tag run-tag))
                                                             :version version}
                                               :throw-exceptions false}))))))

(defn add-pull-request-comment
  "Creates comment and returns the comment id"
  [authed-client {:keys [owner-login repo-slug]} pr-id text]
  (let [url (to-url "projects" owner-login "repos" repo-slug "pull-requests" pr-id "comments")]
    (:id (api-request authed-client :post url {:form-params {:text text}}))))

(defn create-repo-webhook
  "Creates comment and returns a pair [id, url]"
  [authed-client {:keys [owner-login repo-slug]} callback-url secret events]
  (let [url (to-url "projects" owner-login "repos" repo-slug "webhooks")
        {:keys [id]} (api-request authed-client
                                  :post url
                                  {:form-params
                                   {:name "CodeScene Webhook"
                                    :url callback-url
                                    :active true
                                    :configuration {:secret secret}
                                    :events events}})]
    (to-url (request-url authed-client url) id)))

(defn get-repos-webhooks
  [authed-client repo-infos our-hook? concurrency]
  (let [extract (fn [req-url {:keys [id url updatedDate secret events active]}]
                  {:url (str req-url "/" id )
                   :callback-url url
                   :secret secret
                   :events events
                   :enabled? active
                   :timestamp updatedDate})
        run-fn (fn [{:keys [owner-login repo-slug] :as paging-state}]
                 (let [url (to-url "projects" owner-login "repos" repo-slug "webhooks")
                       full-url (request-url authed-client url)
                       paging-state (api-client/with-item-xf paging-state (comp (map #(extract full-url %))
                                                                                (filter our-hook?)))]
                   (api-page-request authed-client paging-state url {})))]
    (page/paginate! (page/async-fn run-fn concurrency) {} repo-infos)))


(defn create-build
  [authed-client commit-id build]
  (let [client (api-client/with-updated-api-url
                 authed-client
                 #(str/replace-first % #"/rest/api/1.0|/?$" "/rest/build-status/1.0"))]
    (api-request client
                 :post
                 (format "/commits/%s" commit-id)
                 {:form-params build})))

(defn create-report
  [authed-client {:keys [owner-login repo-slug sha]} payload annotations?]
  (let [url (to-url "projects" owner-login "repos" repo-slug "commits" sha "reports" (:id payload))
        client (api-client/with-updated-api-url
                 authed-client
                 #(str/replace-first
                    %
                    ;; Yes, you're seeing this right... these aren't under api endpoint at all
                    #"/rest/api/1.0|/?$"
                    "/rest/insights/1.0"))]
    (api-request client :put url {:form-params (dissoc payload :id :annotations)})
    (when annotations?
      (api-request client :delete (str url "/annotations") {}))
    (doseq [:when annotations?
            annotation-batch (partition-all 100 (take 1000 (:annotations payload)))]
      (api-request client
                   :post
                   (str url "/annotations")
                   {:form-params {:annotations annotation-batch}}))))

(defn host->api-url [host]
  (format "https://%s/rest/api/1.0" host))

(comment
  (def token (System/getenv "BITBUCKET_SERVER_TOKEN"))
  (def api-url "http://localhost:7990/rest/api/1.0")
  (def repo "analysis-target"))
