(ns codescene.features.repository-provider.bitbucket.api
  "Bitbucket api access.

  Bitbucket uses codescene.features.components.http connection pool along with a small cache.

  See codescene.features.client.api for refreshing tokens logic."
  (:require [clj-http.client :as http-client]
            [clojure.string :as str]
            [clojure.spec.alpha :as s]
            [codescene.specs :as base-specs]
            [codescene.features.client.api :as api-client]
            [codescene.features.util.string :refer [truncate]]
            [codescene.features.repository-provider.providers :as providers]
            [codescene.features.repository-provider.specs :as specs]
            [codescene.features.util.maps :refer [map-of]]
            [codescene.url.url-utils :as url]
            [evolutionary-metrics.trends.dates :as dates]
            [meta-merge.core :refer [meta-merge]]
            [org.clojars.roklenarcic.paginator :as page]
            [slingshot.slingshot :refer [try+]]
            [taoensso.timbre :as log]))

(def ^:private pagelen 100)

(defn ->scopes [scope-str]
  (some-> scope-str
          (str/split #"\s*,\s*")
          set))

(defn- process-exception
  [exception {:keys [url] :as params}]
  (let [{:keys [body status]} (ex-data exception)
        parsed-body (api-client/json-safe-parse body)
        message (or (get parsed-body :error_description)
                    (get-in parsed-body [:error :message])
                    (ex-message exception))
        token-expired? (some? (re-find #"(?i)expired" (or message "")))
        repo-name (some->> message
                           (re-find #"(?i)repository.*\s+([^\s/]+/[^\s/]+)")
                           second)
        data (map-of message repo-name url)]
    (log/debugf "Bitbucket call returned %s %s" status body)
    (log/info exception "Bitbucket call exception" (api-client/printable-req params))
    (case status
      401 (api-client/ex-unauthorized :bitbucket exception (assoc data :token-expired? token-expired?))
      403 (api-client/ex-forbidden :bitbucket exception data)
      ; currently they return 403 for private repos you cannot access, but how
      ; long before they start returning 404 like GitHub to prevent info leak?
      ; I don't want this to suddenly break because we didn't pay attention to BitBucket API updates
      404 (api-client/ex-forbidden :bitbucket exception data)
      (api-client/ex-http-error :bitbucket exception (or status 0) data))))

(s/fdef api-request*
        :args (s/cat :auth-token ::specs/auth-token
                     :method #{:get :post :delete :put :patch :options :head
                               :trace :connect}
                     :url string?
                     :clj-http-params (s/nilable map?)))
(defn api-request*
  "Core function for API requests. Requires ApiClient token.

  Uses augment-request to add some sort of credentials to the request.
  If request returns 401 then on-expired-token will be called on auth-token.

  If that returns a ApiClient, the request will be retried."
  [authed-client method url request-params]
  (let [params (meta-merge
                 {:url url
                  :as :json
                  :content-type :json
                  :request-method method}
                 request-params)
        final-params (-> authed-client
                         (api-client/augment-request params)
                         (api-client/fix-api-url "https://api.bitbucket.org/2.0"))]
    (when-not (:connection-manager final-params)
      (log/warnf "No connection manager, url=%s, avoid for production use." (:url final-params)))
    (try
      (http-client/request final-params)
      (catch Exception e
        (-> e
            (process-exception final-params)
            (api-client/retry-token authed-client)
            (api-request* method url request-params))))))

(defn api-request [authed-client method url request-params]
  (:body (api-request* authed-client method url request-params)))

(defn api-page-request
  "A helper function for a request that requests a particular page. It is based on a paging state
  and returns an items+cursor map."
  [authed-client {:keys [cursor add-page]} url request-params]
  (api-client/check-interrupt)
  (let [resp (if cursor
               (api-request authed-client :get cursor (dissoc request-params :query-params))
               (api-request authed-client :get url request-params))]
    (add-page (:values resp) (:next resp))))

(defn api-paginated-load
  [authed-client url request-params]
  (-> #(api-page-request authed-client % url request-params)
      page/paginate-one!
      page/unwrap))

(s/fdef get-my-info
        :args (s/cat :auth-token ::specs/auth-token))
(defn get-my-info
  "https://developer.atlassian.com/bitbucket/api/2/reference/resource/user"
  [authed-client]
  (let [fields "uuid,username,display_name,account_id"
        {:keys [body headers]}
        (api-request* authed-client :get "user" {:query-params {"fields" fields}})]
    (assoc body :scope (->scopes (headers "X-OAuth-Scopes")))))

(s/fdef repo-path
        :args (s/cat :repo-info ::specs/repo-info)
        :ret string?)
(defn repo-path
  "Produces a single string that is used in various API paths
  where it represents a path to a particular repository."
  [{:keys [owner-login repo-slug external-id]}]
  (if external-id
    (str "{}/" external-id)
    (str owner-login "/" repo-slug)))

(s/fdef get-my-emails
        :args (s/cat :auth-token ::specs/auth-token))
(defn get-my-emails
  "https://developer.atlassian.com/bitbucket/api/2/reference/resource/user/emails"
  [authed-client]
  (-> (api-request authed-client
                   :get
                   "user/emails"
                   {:query-params {"pagelen" pagelen}})
      :values))

(s/fdef get-repo-main-branch
        :args (s/cat :auth-token ::specs/auth-token
                     :repo-info ::specs/repo-info))
(defn get-repo-main-branch
  "Pull main branch information from the general repo information.

  https://developer.atlassian.com/bitbucket/api/2/reference/resource/repositories/%7Bworkspace%7D/%7Brepo_slug%7D"
  [authed-client repo-info]
  (-> (api-request
        authed-client
        :get
        (format "repositories/%s" (repo-path repo-info))
        {:query-params {"fields" "mainbranch"}})
      (get-in [:mainbranch :name] "master")))

(defn- get-repo-branches*
  [authed-client max-items {:keys [no-branch-fallback? add-page pages repo-slug] :as paging-state}]
  (log/debugf "Getting branches for %s, page: %s" repo-slug (inc pages))
  (if no-branch-fallback?
    (add-page nil)
    (api-page-request
      authed-client
      (-> paging-state
          (api-client/with-item-xf (map :name))
          (api-client/limit-items max-items))
      (format "repositories/%s/refs/branches" (repo-path paging-state))
      {:query-params {"fields" "next,values.name"
                      "pagelen" pagelen}})))

(defn get-repos-branches
  [authed-client repo-infos max-concurrent-requests max-branches]
  (let [get-repo-defaults (page/async-fn (providers/branch-defaults-fn authed-client get-repo-main-branch)
                                         max-concurrent-requests)
        get-branches (partial get-repo-branches* authed-client max-branches)]
    (->> repo-infos
         (mapv get-repo-defaults)
         (mapv page/unwrap-async)
         (page/paginate! (page/async-fn get-branches max-concurrent-requests) {})
         (mapv providers/->branch-results))))

(def repository-list-fields
  (str/join "," ["pagelen"
                 "next"
                 "values.slug"
                 "values.uuid"
                 "values.links.clone"
                 "values.is_private"
                 "values.full_name"
                 "values.workspace.uuid"
                 "values.mainbranch"]))

(s/def ::show-private? boolean?)
(s/fdef get-repos
        :args (s/cat :auth-token ::specs/auth-token
                     :show-private? ::show-private?))
(defn get-repos
  "Returns a list of repos visible by the user or an org."
  [authed-client show-private?]
  (log/debugf "Getting the current user's Bitbucket repos show-private?=%s" show-private?)
  (let [private-query (if show-private? "" " and is_private = false")]
    (api-paginated-load
      authed-client
      "repositories"
      {:query-params {"pagelen" pagelen
                      "fields" repository-list-fields
                      "role" "member"
                      "q" (str "scm = \"git\"" private-query)}})))

(s/fdef get-workspace-repos :args (s/cat :auth-token ::specs/auth-token
                                         :workspace-uuid string?
                                         :show-private? ::show-private?))
(defn get-workspace-repos
  "Returns a list of repos belonging to a workspace."
  [authed-client workspace-uuid show-private?]
  (let [private-query (if show-private? "" " and is_private = false")]
    (log/debugf "Getting Bitbucket repos workspace=%s, show-private?=%s" workspace-uuid show-private?)
    (api-paginated-load
      authed-client
      (format "repositories/%s" workspace-uuid)
      {:query-params {"pagelen" pagelen
                      "fields" repository-list-fields
                      "role" "member"
                      "q" (str "scm = \"git\"" private-query)}})))

(s/fdef has-repo-access?
        :args (s/cat :auth-token ::specs/auth-token
                     :repo-info ::specs/repo-info
                     :access #{:clone :read :write :admin}))
(defn has-repo-access?
  "Checks if user has access to the repo of at least the specified level.

  Returns the actual level of access, nil if the access level is not high enough.

  Levels are :clone, :read, :write, :admin.

  :clone -> repo must be cloneable (so any access will do)
  :read -> explicit read (so called `repository` access, also allows creating commit build status)
  :write -> `repository:write`
  :admin -> `admin`"
  [authed-client {:keys [owner-login repo-slug external-id] :as repo-info} access-level]
  (try+
    (if (= :clone access-level)
      (do
        ; asking for permissions doesn't work on public repos that you haven't been explicitly
        ; given access to, so we check normal access when asking for read access.
        (api-request
          authed-client
          :get
          (format "repositories/%s" (repo-path repo-info))
          {:query-params {"fields" "uuid"}})
        :clone)
      (let [repo-query (if external-id
                         (format "repository.uuid = \"%s\"" external-id)
                         (format "repository.full_name = \"%s/%s\"" owner-login repo-slug))
            params {:query-params {"q" (format "%s and permission>=\"%s\"" repo-query (name access-level))
                                   "fields" "values.permission"}}]
        (-> (api-request authed-client :get "user/permissions/repositories" params)
            (get-in [:values 0 :permission])
            keyword)))
    (catch [:type :http-error] e)))

(s/fdef repo-accessibility
        :args (s/cat :authed-client ::specs/auth-token
                     :commit-info ::specs/repo-info))
(defn repo-accessibility [authed-client repo-info]
  (try+
    (let [fields (str/join "," ["slug" "uuid" "links.clone" "is_private" "full_name"])
          repo (api-request authed-client
                            :get
                            (format "repositories/%s" (repo-path repo-info))
                            {:query-params {"fields" fields}})
          clone-url (->> repo :links :clone (some #(when (= "https" (:name %)) (:href %))))
          private? (:is_private repo)]
      (if (nil? private?)
        (merge repo-info {:private? true :accessible? false})
        {:provider-id :bitbucket
         :external-id (:uuid repo)
         :owner-login (->> repo :full_name (re-find #"^[^/]+"))
         :repo-slug (:slug repo)
         :url (-> clone-url
                  (url/repo-url->repo-info :bitbucket false)
                  providers/repo-info->html-url
                  (str ".git"))
         :private? private?
         :accessible? true}))
    (catch [:type :http-error :status 401] _
      (log/errorf (:throwable &throw-context) "Error getting repository accessibility: %s" (pr-str repo-info))
      repo-info)
    (catch [:type :http-error :status 403] _
      (log/debugf (:throwable &throw-context) "Could not access repository %s. Marking as inaccessible." repo-info)
      (merge repo-info {:private? true :accessible? false}))))


(s/fdef get-user-workspaces
        :args (s/cat :auth-token ::specs/auth-token))
(defn get-user-workspaces
  [authed-client]
  (log/debugf "Getting Bitbucket user's workspaces")
  (api-paginated-load
    authed-client
    "user/permissions/workspaces"
    {:query-params {"pagelen" pagelen}}))

(s/fdef get-workspace-members
        :args (s/cat :auth-token ::specs/auth-token
                     :org-login string?))
(defn get-workspace-members
  [authed-client org-login]
  (log/debugf "Getting Bitbucket workspace members")
  (let [fields (str/join "," ["pagelen" "next" "values.user"])]
    (api-paginated-load
      authed-client
      (format "workspaces/%s/members" org-login)
      {:query-params {"pagelen" pagelen
                      "fields" fields}})))

(s/fdef get-workspace
        :args (s/cat :auth-token ::specs/auth-token
                     :workspace-slug string?))
(defn get-workspace
  [authed-client workspace-slug]
  (log/debugf "Getting Bitbucket workspace %s" workspace-slug)
  (api-request
    authed-client
    :get
    (format "workspaces/%s" workspace-slug)
    {}))

(s/fdef try-get-workspace-uuid
        :args (s/cat :auth-token ::specs/auth-token
                     :owner-login ::base-specs/non-empty-string))
(defn try-get-workspace-uuid
  [authed-client owner-login]
  (try+
    (-> (api-request
          authed-client
          :get
          (format "workspaces/%s" owner-login)
          {:query-params {"fields" "uuid"}})
        :uuid)
    (catch [:type :http-error] e)))

(s/fdef get-pr-commit-shas
        :args (s/cat :auth-token ::specs/auth-token
                     :repo-info ::specs/repo-info
                     :pr-id pos-int?))
(defn get-pr-commit-shas
  "Returns list of commit shas on the pull request of specified repo, ordered
   from oldest to newest."
  [authed-client repo-info pr-id]
  (log/debugf "Getting the Bitbucket PR commits %s PR#%d"
              (repo-path repo-info) pr-id)
  (let [fields (str/join "," ["pagelen"
                              "next"
                              "values.hash"])
        commits (api-paginated-load
                  authed-client
                  (format "repositories/%s/pullrequests/%d/commits" (repo-path repo-info) pr-id)
                  {:query-params {"pagelen" pagelen
                                  "fields" fields}})]
    (vec (rseq (mapv :hash commits)))))

(s/fdef add-webhook
        :args (s/cat :auth-token ::specs/auth-token
                     :repo-info ::specs/repo-info
                     :full-callback-url ::base-specs/non-empty-string
                     :events (s/coll-of ::base-specs/non-empty-string :min-count 1)))
(defn add-webhook
  [authed-client repo-info full-callback-url events]
  (log/debugf "Adding webhook to %s" (repo-path repo-info))
  (let [hook (api-request
               authed-client
               :post
               (format "/repositories/%s/hooks" (repo-path repo-info))
               {:form-params {:description "CodeScene Webhook"
                              :url full-callback-url
                              :active true
                              :events events}})]
    (-> hook :links :self :href)))

(defn get-repos-webhooks
  [authed-client repo-infos our-hook? concurrency]
  (let [extract (fn [{:keys [links url created_at events active]}]
                  ;; TODO ROK external id?
                  {                                         ;:external-id (->external-id account (:projectId publisherInputs) (:repository publisherInputs))
                   :url (-> links :self :href)
                   :callback-url url
                   :events events
                   :enabled? active
                   :timestamp created_at})
        run-fn (fn [paging-state]
                 (api-page-request
                   authed-client
                   (api-client/with-item-xf paging-state (comp (map extract)
                                                               (filter our-hook?)))
                   (format "/repositories/%s/hooks" (repo-path paging-state))
                   {}))]
    (page/paginate! (page/async-fn run-fn concurrency) {} repo-infos)))

(s/def :bitbucket-build/name ::base-specs/non-empty-string)
(s/def :bitbucket-build/url ::base-specs/uri)
(s/def :bitbucket-build/key ::base-specs/non-empty-string)
(s/def :bitbucket-build/description ::base-specs/non-empty-string)
(s/def :bitbucket-build/state #{"INPROGRESS" "STOPPED" "FAILED" "SUCCESSFUL"})
(s/def ::build-status
  (s/keys :opt-un [:bitbucket-build/name :bitbucket-build/description :bitbucket-build/url
                   :bitbucket-build/key :bitbucket-build/state]))

(s/def :bitbucket-report/id ::base-specs/non-empty-string)
(s/def :bitbucket-report/title ::base-specs/non-empty-string)
(s/def :bitbucket-report/details ::base-specs/non-empty-string)
(s/def :bitbucket-report/report_type #{"SECURITY" "COVERAGE" "TEST" "BUG"})
(s/def :bitbucket-report/reporter ::base-specs/non-empty-string)
(s/def :bitbucket-report/link ::base-specs/uri)
(s/def :bitbucket-report/logo_url ::base-specs/uri)
(s/def :bitbucket-report/result #{"PASSED" "FAILED" "PENDING"})
(s/def ::report-data
  (s/keys :req-un [:bitbucket-report/id :bitbucket-report/title :bitbucket-report/details
                   :bitbucket-report/report_type ]
          :opt-un [:bitbucket-report/result :bitbucket-report/reporter :bitbucket-report/link :bitbucket-report/logo_url]))

(s/fdef create-commit-build
        :args (s/cat :auth-token ::specs/auth-token
                     :commit-info ::specs/commit-info
                     :build ::build-status))
(defn create-commit-build
  "https://developer.atlassian.com/bitbucket/api/2/reference/resource/repositories/%7Bworkspace%7D/%7Brepo_slug%7D/commit/%7Bnode%7D/statuses/build"
  [authed-client {:keys [sha] :as commit-info} build-data]
  (api-request
    authed-client
    :post
    (format "repositories/%s/commit/%s/statuses/build"
            (repo-path commit-info)
            sha)
    {:form-params build-data}))

(s/fdef get-commit-build-by-url
        :args (s/cat :auth-token ::specs/auth-token
                     :url ::base-specs/uri))
(defn get-commit-build-by-url
  [authed-client url]
  (api-request authed-client :get url {}))

(s/fdef update-commit-build-by-url
        :args (s/cat :auth-token ::specs/auth-token
                     :url ::base-specs/uri
                     :build ::build-status))
(defn update-commit-build-by-url
  [authed-client url build-data]
  (api-request
    authed-client
    :put
    url
    {:form-params build-data}))

(s/fdef add-pull-request-comment
        :args (s/cat :auth-token ::specs/auth-token
                     :repo-info ::specs/repo-info
                     :pr-id pos-int?
                     :content ::base-specs/non-empty-string))
(defn add-pull-request-comment
  "https://developer.atlassian.com/bitbucket/api/2/reference/resource/repositories/%7Bworkspace%7D/%7Brepo_slug%7D/pullrequests/%7Bpull_request_id%7D/comments#post"
  [authed-client repo-info pr-id content]
  (api-request
    authed-client
    :post
    (format "repositories/%s/pullrequests/%s/comments"
            (repo-path repo-info)
            pr-id)
    {:form-params {:content {:raw content}}}))

(s/fdef delete-pull-request-comment
        :args (s/cat :auth-token ::specs/auth-token
                     :comment map?))
(defn delete-pull-request-comment
  "Use link on comment object to delete the comment."
  [authed-client comment-obj]
  ;; for some reason the self-href looks like this:
  ;; https://bitbucket.org/!api/2.0/repositories/RokLenarcic/analysis-target/pullrequests/4/comments/292911079
  ;; but we want https://api.bitbucket.org/2.0/repositories/RokLenarcic/analysis-target/pullrequests/4/comments/292911079
  ;; the first URL will work with GET, but DELETE will throw error about the credentials
  (let [url (str/replace-first (-> comment-obj :links :self :href) #"^http.*/!api/2.0" "")]
    (log/infof "Remove old Bitbucket Comment with url %s" url)
    (api-request authed-client :delete url {})))

(defn principal-uuid
  [authed-client]
  ;; Some customers Apps aren't updated with "account" scope, so grab tenant directly
  (or (-> authed-client :delegate :tenant)
      (:uuid (try+ (get-my-info authed-client)
                   ;; onprem tokens dont have account scope
                   (catch [:status 403] _ nil)))))

(s/fdef get-pull-request-comments
        :args (s/cat :auth-token ::specs/auth-token
                     :repo-info ::specs/repo-info
                     :pr-id pos-int?
                     :content ::base-specs/non-empty-string))
(defn get-pull-request-comments
  "https://developer.atlassian.com/bitbucket/api/2/reference/resource/repositories/%7Bworkspace%7D/%7Brepo_slug%7D/pullrequests/%7Bpull_request_id%7D/comments#get"
  [authed-client repo-info pr-id content]
  (api-request
    authed-client
    :get
    (format "repositories/%s/pullrequests/%s/comments"
            (repo-path repo-info)
            pr-id)
    {:query-params {"pagelen" pagelen
                    "q" (str "content.raw ~ \""
                             content
                             (if-let [uuid (principal-uuid authed-client)]
                               (str "\" and user.uuid = \"" uuid)
                               ;; onprem tokens dont have account scope, disable user filter
                               "")
                             "\" and deleted = false")}}))


(defn delete-comments-with-runtag
  "Deletes PR comments that contain the run-tag."
  [authed-client repo-info pr-id run-tag]
  (let [comments (get-pull-request-comments authed-client repo-info pr-id run-tag)]
    (doseq [comment (:values comments)]
      (delete-pull-request-comment authed-client comment))))

(s/fdef put-commit-report
        :args (s/cat :auth-token ::specs/auth-token
                     :commit-info ::specs/commit-info
                     :report-data ::report-data
                     :app-key ::base-specs/non-empty-string))
(defn put-commit-report
  "https://developer.atlassian.com/bitbucket/api/2/reference/resource/repositories/%7Bworkspace%7D/%7Brepo_slug%7D/commit/%7Bcommit%7D/reports/%7BreportId%7D#put"
  [authed-client {:keys [sha] :as repo-info} report-data bitbucket-app-key]
  (log/debugf "Putting commit report repo-info=%s report=%s" repo-info report-data)
  (api-request
    authed-client
    :put
    (format "repositories/%s/commit/%s/reports/%s" (repo-path repo-info) sha bitbucket-app-key)
    {:form-params report-data}))

(s/fdef get-commit-report
        :args (s/cat :auth-token ::specs/auth-token
                     :commit-info ::specs/commit-info
                     :app-key ::base-specs/non-empty-string))
(defn get-commit-report
  [authed-client {:keys [sha] :as repo-info} report-id]
  (api-request authed-client
               :get
               (format "repositories/%s/commit/%s/reports/%s" (repo-path repo-info) sha report-id)
               {}))

(s/fdef get-commit-sha
        :args (s/cat :auth-token ::specs/auth-token
                     :repo-info ::specs/repo-info
                     :commit ::specs/sha))
(defn get-commit-sha
  "Convert short sha into long one."
  [authed-client repo-info commit]
  (-> authed-client
      (api-request :get
                   (format "repositories/%s/commit/%s" (repo-path repo-info) commit)
                   {})
      :hash))

(defn add-annotations [authed-client commit-info report-id annotations]
  (let [fix-data #(-> % (update :summary truncate 450))]
    (api-request authed-client
                 :post
                 (format "repositories/%s/commit/%s/reports/%s/annotations"
                         (repo-path commit-info)
                         (:sha commit-info)
                         report-id)
                 {:form-params (mapv fix-data annotations)})))

(s/fdef get-pull-requests
  :args (s/cat :auth-token ::specs/auth-token
               :repo-info ::specs/repo-info
               :search-options map?))
(defn get-pull-requests
  "Fetch pull requests"
  [authed-client repo-info {:keys [state since] :as _search-options}]
  (let [clauses (filter some?
                        [(when state (format "state = \"%s\"" state))
                         (when since (format "updated_on > %s" since))])
        ;; paging size 100 not allowed for some reason
        query-params (merge {"pagelen" 50
                             "sort-" "updated_on"}
                            (when (seq clauses)
                              {"q" (str/join " AND " clauses)}))]
    (api-paginated-load
      authed-client
      (format "repositories/%s/pullrequests" (repo-path repo-info))
      {:query-params query-params})))

(defn host->api-url [host]
  (when (and host (not= host "bitbucket.org"))
    (format "https://%s/api/2.0" host)))

(comment
  (def user (System/getenv "BITBUCKET_USER"))
  (def password (System/getenv "BITBUCKET_APP_PASSWORD"))
  (def repo-info (codescene.url.url-utils/repo-url->repo-info "git@bitbucket.org:empear/analysis-target.git" :bitbucket true))
  (def authed-client [user password])
  (get-pull-requests authed-client repo-info {:since (dates/string->date "2021-12-01")
                                              :state "MERGED"})
  )
