(ns codescene.features.reports.pdf-reports
  "API for generating pdf-reports based on spec'd report data.
  The report data is a map with entries corresponding to sections in the report,
  where a nil entry means to exclude that section from the report."
  (:refer-clojure :exclude [chunk])
  (:require [clojure.java.io :as io]
            [clojure.spec.alpha :as s]
            [clojure.string :as str]
            [clojure.walk :as walk]
            [clj-time.local :as tl]
            [clj-time.format :as tf]
            [clj-pdf.core :as pc]
            [codescene.features.reports.pdf-reports-def :as pdf-reports-def]
            [codescene.features.reports.pdf-helpers :as ph]
            [codescene.features.reports.util :as reports-util]
            [codescene.presentation.health :as health]
            [codescene.presentation.system-health :as system-health]
            [codescene.features.reports.pdf-reports-spec :as reports-spec])
  (:import [java.awt Color]
           [com.lowagie.text.pdf PdfContentByte]
           [com.lowagie.text Rectangle Image]
           [java.net URL]))

(defn- date-as-string [date]
  (tf/unparse (:year-month-day tf/formatters) date))

(defn- date-from-string [date]
  (tf/parse (:year-month-day tf/formatters) date))

(defn- current-date-as-string []
  (date-as-string (tl/local-now)))

(defn- nilable->time [date]
  (if (some? date)
    (str (tf/unparse (:year-month-day tf/formatters) date) " " (tf/unparse (:hour-minute tf/formatters) date))
    "-"))

(defn- nilable->percentage-str
  ([fraction]
   (nilable->percentage-str fraction 0))
  ([fraction decimals]
   (if (some? fraction)
     (str (Math/round (* 100 fraction)) "%")
     "-")))

(defn- nilable->str [value]
  (if (some? value) (str value) "-"))

(defn split-first [re s]
  (str/split s re 2))

(defn- split-last [re s]
  (let [pattern (re-pattern (str re "(?!.*" re ")"))]
    (split-first pattern s)))


(defn- ^URL load-png-resource [dir name]
  (let [path (str dir "/" name ".png")]
    (io/resource path)))

(defn- code-health->color [code-health]
  (cond (<= code-health pdf-reports-def/code-health-low)
        pdf-reports-def/red
        (>= code-health pdf-reports-def/code-health-high)
        pdf-reports-def/green
        :else pdf-reports-def/yellow))

(defn- code-health->cell-style [code-health]
  (if (nil? code-health)
    :code-health-unset-cell
    (cond
      (<= code-health pdf-reports-def/code-health-low) :code-health-alarm-cell
      (>= code-health pdf-reports-def/code-health-high) :code-health-ok-cell
      :else :code-health-warning-cell)))

(defn- code-health->trend-image [code-health last-code-health]
  (if (or (nil? code-health) (nil? last-code-health))
    "not-calculated"
    (let [diff (- ^Integer code-health ^Integer last-code-health)]
      (cond (>= diff pdf-reports-def/code-health-trend-high) "big-up"
            (>= diff pdf-reports-def/code-health-trend-low) "medium-up"
            (> diff 0) "small-up"
            (<= diff (- pdf-reports-def/code-health-trend-high)) "big-down"
            (<= diff (- pdf-reports-def/code-health-trend-low)) "medium-down"
            (< diff 0) "small-down"
            :else "stable"))))

(defn- code-health->image-fn [img-dir code-health last-code-health]
  (let [trend-image-name (code-health->trend-image code-health last-code-health)
        trend-image-resource (load-png-resource img-dir (str trend-image-name "-graph"))
        trend-image-good-resource (load-png-resource img-dir (str trend-image-name "-good-graph"))
        trend-image-fn (ph/fit-image-fn :center :bottom trend-image-resource)
        trend-image-good-fn (ph/fit-image-fn :center :bottom trend-image-good-resource)]
    (fn [^Rectangle pos ^PdfContentByte canvas]
      (when trend-image-resource (trend-image-fn pos canvas))
      (when trend-image-good-resource (trend-image-good-fn pos canvas)))))

(defn- team-autonomy->cell-style [team-autonomy]
  (case team-autonomy
    :high-team-autonomy :team-autonomy-alarm-cell
    :medium-team-autonomy :team-autonomy-neutral-cell
    :low-team-autonomy :team-autonomy-ok-cell
    :team-autonomy-unset-cell))

(defn- team-autonomy->str [team-autonomy]
  (case team-autonomy
    :high-team-autonomy "High"
    :medium-team-autonomy "Medium"
    :low-team-autonomy "Low"
    "-"))

(defn- hotspot-severity->color [severity]
  (case severity
    3 :hotspot-severity-alarm
    2 :hotspot-severity-warning
    :hotspot-severity-ok))

(defn- system-mastery->cell-style [system-mastery]
  (if (nil? system-mastery)
    :system-mastery-unset-cell
    (cond
      (>= system-mastery pdf-reports-def/system-mastery-high) :system-mastery-ok-cell
      (< system-mastery pdf-reports-def/system-mastery-low) :system-mastery-alarm-cell
      :else :system-mastery-warning-cell)))

(defn- delivery-risk->cell-style [delivery-risk]
  (if (nil? delivery-risk)
    :delivery-risk-unset-cell
    (cond
      (>= delivery-risk pdf-reports-def/delivery-risk-high) :delivery-risk-alarm-cell
      (< delivery-risk pdf-reports-def/delivery-risk-low) :delivery-risk-ok-cell
      :else :delivery-risk-warning-cell)))

(defn- missed-goals->cell-style [missed-goals]
  (if (nil? missed-goals)
    :missed-goals-unset-cell
    (if (zero? missed-goals)
      :missed-goals-ok-cell
      :missed-goals-alarm-cell)))

(defn- total-goals->str [total-goals]
  (if (some? total-goals) (str "/" (nilable->str total-goals) "")))

(defn- defects->cell-style [defects last-defects]
  ;; TODO: Colors
  (if (or (nil? defects) (nil? last-defects))
    :defects-unset-cell
    :defects-neutral-cell))

(defn- defects->trend-image [defects last-defects]
  (if (or (nil? defects) (nil? last-defects))
    "not-calculated"
    (let [diff (- ^Integer defects ^Integer last-defects)]
      (cond (> diff 0) "up-and-worse"
            (< diff 0) "down-and-better"
            :else "no-image"))))

(defn- system-code-health->trend-arrow-fn
  [img-dir trend status [lef-pos top-pos width height]]
  (fn [^Rectangle pos ^PdfContentByte canvas]
    (let [name (if (or (nil? trend) (nil? status))
               "not-calculated"
               (str trend "-" status))
          img-url (load-png-resource (str img-dir "/codehealth") name)
          img (when img-url (Image/getInstance img-url))]
    (when img
      (do
        (.scaleToFit img width width)
        (.setAbsolutePosition img (- (.getLeft pos) lef-pos) (- (.getTop pos) top-pos (/ (.getHeight pos) 2) (/ width 2)) )
        (.addImage canvas img))))))

(defn- defects->trend-image-fn [img-dir defects last-defects]
  (when-let [trend-image (load-png-resource img-dir (defects->trend-image defects last-defects))]
    (ph/fit-image-fn :right :center trend-image)))


(defn- commits-last-month->activity-image [commits-last-month max-commits-last-month]
  (let [activity-percentage (when (and commits-last-month max-commits-last-month (> max-commits-last-month 0))
                              (int (* 100 (double (/ commits-last-month max-commits-last-month)))))]
    (cond
      (nil? activity-percentage) "not-calculated"
      (<= activity-percentage 20) "very-cold"
      (<= activity-percentage 40) "cold"
      (<= activity-percentage 60) "warm"
      (<= activity-percentage 80) "very-warm"
      :else "hot")))

(defn- system-code-health->cell-style
  [current-score-status]
  (case (keyword current-score-status)
    :good :system-code-health-good
    :normal :system-code-health-normal
    :bad :system-code-health-bad
    ""))

(defn- overall-code-health->cell-style
  [current-score-status]
  (case (keyword current-score-status)
    :good :overall-code-health-good
    :normal :overall-code-health-normal
    :bad :overall-code-health-bad
    ""))

(defn- sub-system-spacing-row [count]
  (mapv (fn [_] (ph/cell [:sub-system-spacing-cell] ""))
        (range count)))

(defn- sub-system-health-row [sub-system options]
  (let [{:keys [sub-system team-autonomy system-mastery code-health last-code-health defects last-defects delivery-risk]} sub-system
        {:keys [img-dir]} options]
    (ph/row
      (ph/cell [:sub-system-name-cell]
               (ph/paragraph [:sub-system-name] sub-system))
      (ph/cell [:sub-system-team-autonomy-cell (team-autonomy->cell-style system-mastery)]
               (ph/paragraph [:sub-system-team-autonomy] (team-autonomy->str team-autonomy)))
      (ph/cell [:sub-system-mastery-cell (system-mastery->cell-style system-mastery)]
               (ph/paragraph [:sub-system-mastery] (nilable->percentage-str system-mastery)))
      (ph/cell [:sub-system-code-health-cell (system-code-health->cell-style code-health)]
               {:background-layer-fn (code-health->image-fn img-dir code-health last-code-health)}
               (ph/paragraph [:sub-system-code-health] (nilable->str code-health)))
      (ph/cell [:sub-system-defects-cell (defects->cell-style defects last-defects)]
               {:background-layer-fn (defects->trend-image-fn img-dir defects last-defects)}
               (ph/paragraph [:sub-system-defects] (nilable->str defects)))
      (ph/cell [:sub-system-delivery-risk-cell (delivery-risk->cell-style delivery-risk)]
               (ph/paragraph [:sub-system-delivery-risk] (nilable->str delivery-risk))))))

(defn- system-code-health-row [sub-system options]
  (let [{:keys [name team-autonomy system-mastery current-score revisions score-trend current-score-status]} sub-system
        {:keys [img-dir]} options]
    (ph/row
      (ph/cell [:system-code-health-name-cell]
               (ph/paragraph [:system-code-health-name] name))
      (ph/cell [:system-code-health-commits-cell]
               (ph/paragraph [:system-code-health-commits] revisions))
      (ph/cell [:system-code-health-team-autonomy-cell]
               (ph/paragraph [:system-code-health-team-autonomy] (team-autonomy->str team-autonomy)))
      (ph/cell [:system-code-health-system-mastery-cell]
               (ph/paragraph [:system-code-health-system-mastery] (str system-mastery "%")))
      (ph/cell [:system-code-health-cell]
               {:background-layer-fn (system-code-health->trend-arrow-fn img-dir score-trend current-score-status [-5 -2 10 10])}
               (ph/paragraph [:system-code-health (system-code-health->cell-style current-score-status)] (nilable->str current-score))))))

(defn- sub-system-health-table [sub-systems options]
  (let [columns ["Sub-System" "Team Autonomy" "System Mastery" "Code Health" "Defects" "Delivery Risks"]]
    (concat
      (ph/table [:sub-system-health-table] [4 1 1 1 1 1])
      [(ph/cells [:sub-system-health-header-cell] [:sub-system-health-header] columns)]
      (interpose (sub-system-spacing-row 6)
                 (map #(sub-system-health-row % options) sub-systems)))))

(defn- system-code-health-table [sub-systems options]
  (let [columns ["Sub-System" "Commits" "Team Autonomy" "System Mastery" "Code Health"]]
    (reports-util/concatv
      (ph/table [:system-code-health-table] [4 1 1 1 1])
      [(ph/cells [:system-code-health-header-cell] [:system-code-health-header] columns)]
      (interpose (sub-system-spacing-row 5)
                 (map #(system-code-health-row % options) sub-systems)))))

(defn- code-health-doughnut [now]
  (if (number? now)
    (let [code-health-fraction (/ now 10)
          color (code-health->color now)]
      (ph/cell [:overall-code-health-table-cell]
               {:background-layer-fn (ph/draw-using-g2d-fn (reports-util/doughnut-fn code-health-fraction {:fill-color color :show-percentage false}))}
               ""))
    (ph/cell [:overall-code-health-table-cell] (ph/paragraph [] "No data"))))

(defn- overall-code-health-trend [now last-month]
  (if (and (number? now) (number? last-month))
    (cond (= now last-month)
          "No change since last month"
          (> now last-month)
          (str "Up from " last-month " last month")
          (< now last-month)
          (str "Down from " last-month " last month"))))

(defn- overall-code-health-table [dashboard options]
  (let [{:keys [biomarkercurrent
                biomarkermonthback
                full-code-health-average-now
                full-code-health-average-last-month
                full-code-health-worst-performer-now
                full-code-health-worst-performer-last-month]} dashboard
        curent-score-status (health/val->status biomarkercurrent health/health-threshold-for-red health/health-threshold-for-yellow)]
    (reports-util/concatv
      (ph/table [:overall-code-health-table] [45 45 45])
      [(ph/row
        (ph/cell [:overall-code-health-table-cell :overall-code-health-text] "Hotspots")
        (ph/cell [:overall-code-health-table-cell :overall-code-health-text] "Average")
        (ph/cell [:overall-code-health-table-cell :overall-code-health-text] "Worst performer"))
       (ph/row
         (code-health-doughnut biomarkercurrent)
         (code-health-doughnut full-code-health-average-now)
         (code-health-doughnut full-code-health-worst-performer-now))
       (ph/row
         (ph/cell [:overall-code-health-table-cell]
                  (ph/paragraph [:overall-code-health-value] biomarkercurrent)
                  (ph/paragraph [] (overall-code-health-trend biomarkercurrent biomarkermonthback)))
         (ph/cell [:overall-code-health-table-cell]
                  (ph/paragraph [:overall-code-health-value] full-code-health-average-now)
                  (ph/paragraph [] (overall-code-health-trend full-code-health-average-now full-code-health-average-last-month)))
         (ph/cell [:overall-code-health-table-cell]
                  (ph/paragraph [:overall-code-health-value] full-code-health-worst-performer-now)
                  (ph/paragraph [] (overall-code-health-trend full-code-health-worst-performer-now full-code-health-worst-performer-last-month))))])))

(defn- hotspot-detail-paragraph [detail]
  (let [{:keys [description severity]} detail]
    (ph/paragraph [:hotspot-details] (ph/chunk [(hotspot-severity->color severity)] "• ") description)))

(defn- hotspot-row [{:keys [file code-health last-month-code-health last-year-code-health hotspot-details goals] :as _hotspot} options]
  (let [parts (split-last #"/" file)
        file-name (second parts)
        dir-name (first parts)
        {:keys [img-dir]} options
        code-health-values [code-health last-month-code-health last-year-code-health]
        current-score-trend (system-health/get-temporal-scope-for code-health-values "year")
        current-score-status (health/val->status code-health health/health-threshold-for-red health/health-threshold-for-yellow)]
    (ph/row
      (ph/cell [:hotspot-name-cell]
               (ph/paragraph [:hotspot-file-name] file-name)
               (ph/paragraph [:hotspot-dir-name] dir-name))
      (ph/cell [:hotspot-code-health-cell]
               {:background-layer-fn (system-code-health->trend-arrow-fn img-dir current-score-trend current-score-status [-15 -2 10 10])}
               (ph/paragraph [:hotspot-code-health (system-code-health->cell-style current-score-status)] (nilable->str code-health)))
      (apply ph/cell [:hotspot-details-cell]
             (map hotspot-detail-paragraph hotspot-details))
      (ph/cell [:hotspot-goals-cell]
               (ph/paragraph [:hotspot-goals] (nilable->str goals))))))

(defn- hotspots-table [hotspots options]
  (let [columns ["Hotspot Name" "Code Health" "Identified Code Health Issue" "Goals"]]
    (reports-util/concatv
      (ph/table [:hotspots-table] [35 15 30 20])
      [(ph/cells [:hotspots-table-header-cell] [:hotspots-table-header] columns)]
      (interpose (sub-system-spacing-row 4)
        (map #(hotspot-row % options) hotspots)))))


(defn- title-page-graphics [{:keys [title-decoration-1 title-decoration-2]
                             :or
                             {title-decoration-1 [248 238 113]
                              title-decoration-2 [235 73 80]}} g2d]
  (let [[r1 g1 b1] title-decoration-1
        [r2 g2 b2] title-decoration-2]
  (doto g2d
    ; Coordinates for a4, 72 dpi (595x842)
    (.setColor (Color. r1 g1 b1))
    (.fillRect (int 384) (int 0) (int 595) (int 240))
    (.setColor (Color. r2 g2 b2))
    (.fillRect (int 0) (int 569) (int 151) (int 842)))))



(defn- title-page [options]
  (let [{:keys [title date colors] :or {title "New Report" date (current-date-as-string)}} options]
    [[:spacer 13]
     (ph/graphics [] {:under true} (partial title-page-graphics colors))
     (ph/paragraph [:title] title)
     (ph/paragraph [:subtitle] "Behavioral Code Analysis")
     (ph/paragraph [:subtitle] date)
     (ph/paragraph [:subtitle] "by CodeScene")
     (ph/pagebreak)]))

(defn- project-summary-code-health-cell
  [img-dir code-health-now code-health-last-month]
  (let [current-score-trend (system-health/get-temporal-scope-for [code-health-now code-health-last-month] "month")
        current-score-status (health/val->status code-health-now health/health-threshold-for-red health/health-threshold-for-yellow)]
    (ph/cell [:project-code-health-cell]
             {:background-layer-fn (system-code-health->trend-arrow-fn img-dir current-score-trend current-score-status [0 0 10 10])}
             (ph/paragraph [:project-code-health (system-code-health->cell-style current-score-status)] (nilable->str code-health-now)))))

(defn- project-summary-row [project-name max-commits-last-month system-health code-health-overall dashboard-url options]
  (let [{:keys [abandoned-code active-authors analysis-time commits-last-month _former-developers
                lines-of-code missed-goals]} system-health
        {:keys [biomarkercurrent biomarkermonthback full-code-health-average-now full-code-health-average-last-month
                full-code-health-worst-performer-now full-code-health-worst-performer-last-month]} code-health-overall
        {:keys [img-dir]} options
        system-mastery (when abandoned-code (- 1.0 abandoned-code))
        activity-image-name (commits-last-month->activity-image commits-last-month max-commits-last-month)]
    [(ph/table [:project-table] [1 5 5 7]
               (ph/row
                 (ph/cell []
                          (if-let [activity-image (load-png-resource img-dir activity-image-name)]
                            (ph/image [:project-activity-image] activity-image)
                            ""))
                 (ph/cell []
                          (ph/table [:project-description-and-status-table] [1]
                                    (ph/row
                                      (ph/cell [:project-name-cell]
                                               (ph/paragraph [:project-name] (ph/anchor [:project-name :component-link] {:target dashboard-url} project-name))))
                                    (ph/row
                                      (ph/cell [:project-commits-last-month-cell]
                                               (ph/paragraph [:project-commits-last-month] (str (nilable->str commits-last-month) " commits in the last month"))))))
                 (ph/cell []
                          (ph/table [:project-data-table] [1]
                                    (ph/row
                                      (ph/cell [:project-analysis-time-cell]
                                               (ph/paragraph [:project-analysis-time] (ph/chunk [:project-table-label] "Last analysis:  ") (nilable->time analysis-time))))
                                    (ph/row
                                      (ph/cell [:project-lines-of-code-cell]
                                               (ph/paragraph [:project-lines-of-code] (ph/chunk [:project-table-label] "Lines of code:  ") (nilable->str lines-of-code))))
                                    (ph/row
                                      (ph/cell [:project-active-developers-cell]
                                               (ph/paragraph [:project-active-developers] (ph/chunk [:project-table-label] "Active developers:  ") (nilable->str active-authors))))
                                    (ph/row
                                      (ph/cell [:project-lines-of-code-cell]
                                               (ph/paragraph [:project-lines-of-code] (ph/chunk [:project-table-label] "System mastery:  ") (nilable->percentage-str system-mastery))))
                                    (ph/row
                                      (ph/cell [:project-lines-of-code-cell]
                                               (ph/paragraph [:project-lines-of-code] (ph/chunk [:project-table-label] "Missed goals:  ") (nilable->str missed-goals))))))
                                    
                 (ph/cell []
                          (ph/table [:project-health-table] [1 1 1]
                                    (ph/row
                                      (ph/cell [:project-code-health-label] (ph/paragraph [] "Hotspots"))
                                      (ph/cell [:project-code-health-label] (ph/paragraph [] "Average"))
                                      (ph/cell [:project-code-health-label] (ph/paragraph [] "Worst performer")))
                                    (ph/row
                                     (project-summary-code-health-cell img-dir biomarkercurrent biomarkermonthback)
                                     (project-summary-code-health-cell img-dir full-code-health-average-now full-code-health-average-last-month)
                                     (project-summary-code-health-cell img-dir full-code-health-worst-performer-now full-code-health-worst-performer-last-month))))))]))

(defn- early-warning->cell-style
  [level]
  (case (keyword level)
    :info :early-warning-info
    :warning :early-warning-warning
    :danger :early-warning-danger
    ""))

(defn- early-warning-raw [early-warning]
  (ph/row
    (ph/cell [:early-warning-parent-cell]
             (ph/table [] [1]
                       (ph/row
                         (ph/cell [:early-warning-cell :early-warning-cell-bottom-border (early-warning->cell-style (:category early-warning))]
                                  (ph/paragraph [:early-warning-category (early-warning->cell-style (:category early-warning))] (str (str/capitalize (:category early-warning)) " Level Notifications"))))
                       (ph/row
                         (ph/cell [:early-warning-cell]
                                  (ph/paragraph [:early-warning-description] (str (:description early-warning) " - ")
                                                (ph/anchor [:early-warning-link] {:target (:link early-warning)} "View"))))
                       (ph/row
                         (ph/cell [:early-warning-cell]
                                  (ph/paragraph [:early-warning-explanation] (:explanation early-warning))))))))
                       

(defn- early-warnings-section [early-warnings]
  (when (some? early-warnings)
    (concat
      [(ph/paragraph [:heading] "Early Warnings")]
      (if (seq early-warnings)
        [(reports-util/concatv
           (ph/table [:early-warnings-table] [1])
           (interpose (sub-system-spacing-row 1)
                      (map early-warning-raw early-warnings)))]
        [(ph/paragraph [:body] "No early warnings have been detected.")])
      [(ph/pagebreak)])))

(defn- subsystem-health-section [subsystem-health options]
  (when (some? subsystem-health)
    [(ph/paragraph [:heading] "System Health and Risks")
     (if (seq subsystem-health)
       (sub-system-health-table subsystem-health options)
       (ph/paragraph [:body] "No subsystems have been configured."))
     (ph/pagebreak)]))

(defn- system-hotspots-section [hotspots options]
  (when (some? hotspots)
    [(ph/paragraph [:heading] "Hotspots with Code Health Trends")
     (if (seq hotspots)
       (hotspots-table hotspots options)
       (ph/paragraph [:body] "No hotspots have been found."))
     (ph/pagebreak)]))

(defn- subsystem-hotspots-section [subsystem hotspots options]
  (when (some? hotspots)
    [(ph/paragraph [:heading2] (str "Hotspots For " subsystem))
     (if (seq hotspots)
       (hotspots-table hotspots options)
       (ph/paragraph [:body] "No hotspots have been found."))
     (ph/pagebreak)]))

(defn- hotspots-by-subsystem-section [hotspots-by-subsystem options]
  (when (some? hotspots-by-subsystem)
    (concat
      [(ph/paragraph [:heading] "Hotspots By Sub-System")]
      (if (seq hotspots-by-subsystem)
        (mapcat (fn [[subsystem hotspots]] (subsystem-hotspots-section subsystem hotspots options)) hotspots-by-subsystem)
        [(ph/paragraph [:body] "No subsystems have been configured.")
         (ph/pagebreak)]))))

(defn- add-key-for-value [old-key new-key form]
  (let [value (old-key form)]
    (if (some? value)
      (assoc form new-key value)
      form)))

(defn- pick-last-values [report-data timespan]
  (let [last-value-key (if (= timespan :month) :last-month-code-health :last-year-code-health)]
    (walk/postwalk #(add-key-for-value last-value-key :last-code-health %) report-data)))

(defn- with-optional-metadata [pdf-report-metadata options]
  (if-let [font (:font options)]
    (assoc pdf-report-metadata :font {:encoding :unicode :ttf-name font})
    pdf-report-metadata))

(defn- link-to-overridden-code-health-rules
  [{:keys [analysis-dashboard-url] :as _options}]
  (when analysis-dashboard-url ; backwards compatible analysis lib until codescene-features is merged
    (ph/paragraph [] "The project has overridden rules. "
                  (ph/anchor [:component-link] {:target (str analysis-dashboard-url "/scope/analysis-data/overridden-code-health-rules")}
                             "View the overridden rules in CodeScene"))))

(defn- code-health-summary-section
  []
  [(ph/paragraph [:heading] "Summary")
   (ph/paragraph [] "The Technical Health Overview captures the code health of your hotspots on a file level and on an architectural level in order to get a take-home report representing a snapshot of the technical metrics.")
   (ph/paragraph [] "The document is addressed to the technical leaders that want an overview of the strong and weak parts in the codebase.")
   [:spacer 2]])

(defn- code-health-overall-section
  [{:keys [customized-code-health-rules] :as data}
   code-health
   options]
  [(ph/paragraph [:heading] "Overall code health")
   (overall-code-health-table code-health options)
   [:spacer 1]
   (ph/paragraph [] "Code Health identifies factors known to impact Maintenance costs and Delivery Risks. "
                 (ph/anchor [:component-link] {:target (:analysis-dashboard-url options)} "View dashboard ")
                 "to examine further.")
   [:spacer 1]
   (when customized-code-health-rules
     (link-to-overridden-code-health-rules options))])

(defn- code-health-overall-compact-section
  [code-health options]
  [(ph/paragraph [:heading] "Overall code health")
   (overall-code-health-table code-health options)
   [:spacer 1]])

(defn- system-health-section
  [system-health options]
  (when (not-empty system-health)
    [(ph/paragraph [:heading] "System Level code health")
     (system-code-health-table system-health options)
     (ph/pagebreak)]))

(defn technical-health-overview
  "Generate data for pdf generation of technical-health-overview as specified by clj-pdf"
  [data options]
  (let [{:keys [timespan] :or {timespan :month}} options
        {:keys [subsystem-health system-hotspots hotspots-by-subsystem code-health-overall]} (pick-last-values data timespan)]
    (concat
      [(with-optional-metadata pdf-reports-def/pdf-report-metadata options)]
      (title-page options)
      (code-health-summary-section)
      (code-health-overall-section data code-health-overall options)
      [(ph/pagebreak)]
      (system-health-section subsystem-health options)
      (system-hotspots-section system-hotspots options)
      (hotspots-by-subsystem-section hotspots-by-subsystem options))))

(defn trend-report
  "Generate data for pdf generation of trend report as specified by clj-pdf"
  [data options]
  (let [{:keys [early-warnings code-health-overall]} data]
    (concat
      [(with-optional-metadata pdf-reports-def/pdf-report-metadata options)]
      (title-page options)
      (code-health-overall-section data code-health-overall options)
      (early-warnings-section early-warnings))))

(defn- system-health-key-personnel-table [data _options]
  (let [{:keys [abandoned-code key-personnel key-personnel-ratio]} data]
    (ph/table [:system-health-key-personnel-table] [1 4]
              (ph/row
                (ph/cell []
                         [:spacer 1])
                (ph/cell []
                         [:spacer 1]))
              (ph/row
                (ph/cell [:system-health-key-personnel-image-cell]
                         {:background-layer-fn (ph/draw-using-g2d-fn (reports-util/doughnut-fn key-personnel-ratio))} "")
                (ph/cell [:system-health-key-personnel-cell]
                         (ph/paragraph [:system-health-key-personnel] (str (nilable->percentage-str key-personnel-ratio 1) " of the code written by " key-personnel " developers")))
                )
              (ph/row
                (ph/cell []
                         [:spacer 1])
                (ph/cell []
                         [:spacer 1]))
              (ph/row
                (ph/cell [:system-health-abandoned-code-image-cell]
                         {:background-layer-fn (ph/draw-using-g2d-fn (reports-util/doughnut-fn abandoned-code))} "")
                (ph/cell [:system-health-abandoned-code-cell]
                         (ph/paragraph [:system-health-abandoned-code] (str (nilable->percentage-str abandoned-code 1) " total abandoned code"))))
              )))

(defn- key-personnel-section [data options]
  (when data
    [(ph/paragraph [:heading] "Key Personnel")
     (ph/table [:system-health-table] [1]
               (ph/row
                 (system-health-key-personnel-table data options)))
     (ph/pagebreak)]))

(defn- knowledge-loss-fn
  "return an array a map where key is the style and value is the text to display"
  [file]
  (if (:lossinconclusive file)
    {:loss-inconclusive-color "inconclusive"}
    (let [loss (:loss file)
          mastery-str (nilable->percentage-str loss 0)]
      (if (>= loss 0.5)
        {:knowlegde-red-color mastery-str}
        {:knowlegde-blue-color mastery-str}))))

(defn- file-table-row [file knowledge-fn]
  (ph/row
    (ph/cell [:component-filename-cell]
             (ph/paragraph [:knowledge-file-name] (last (:path file)))
             (ph/paragraph [:knowledge-dir-name] (str/join "/" (drop-last (:path file)))))
    (let [knowledge (knowledge-fn file)]
      (ph/cell [:component-knowledge-cell (first (keys knowledge))] (first (vals knowledge))))))

(defn- files-sorted-by-loss [item]
  (->> (:files item)
       (filter #(> (:loss %) 0))
       (sort-by :loss >)))

(defn- system-mastery-heading [item]
  (format "System Mastery For %s - %s"
          (:component item)
          (nilable->percentage-str (:component-mastery item) 1)))

(defn- component-loss-section [item options]
    (let [number_of_files 10
          files (files-sorted-by-loss item)
          count-files (count files)]
      (when (not-empty files)
        (ph/row
          (ph/cell [:component-cell]
             (ph/paragraph [:heading2] (system-mastery-heading item))
             [:spacer 1]
             (reports-util/concatv
               (ph/table [:knowledge-table] [3 1])
               [(ph/row
                  (ph/cell [:component-header] "Filename")
                  (ph/cell [:component-header] "Loss"))]
               (map #(file-table-row % knowledge-loss-fn) (take number_of_files files))
               (when (> count-files number_of_files)
                 [(ph/row
                  (ph/cell [:component-knowledge-cell]
                           [:spacer 1]
                           (ph/paragraph [] (format "and %d more files - " (- count-files number_of_files))
                                         (ph/anchor [:component-link] {:target (:knowledge-loss-url options)} "View them in CodeScene")))
                  (ph/cell [:component-knowledge-cell] ""))])))))))

(defn- get-risk-aspect-for [file]
  (cond
    (:knowledgeislandcomplexcode file)
    :knowledgeislandcomplexcode
    (:knowledgeisland file)
    :knowledgeisland
    (:complexcodebyformercontributors file)
    :complexcodebyformercontributors
    :else
    :noaspect))

(defn- get-knowledge-risk-style
  "return an array a map where key is the style and value is the text to display"
  [aspect]
  (cond
    (= :knowledgeislandcomplexcode aspect)
    {:knowledge-island-complex-code-risk-color "Knowledge Island in Complex Hotspot"}

    (= :knowledgeisland aspect)
    {:knowledge-island-risk-color "Knowledge Island"}

    (= :complexcodebyformercontributors aspect)
    {:complex-code-by-former-contributors-risk-color "Risk: Complex Code by Former Contributors"}

    :else
    {:default-risk-color "" }))

(defn- get-file-row [file]
  (ph/row
    (ph/cell [:component-filename-cell]
             (ph/paragraph [:knowledge-file-name] (last (:path file)))
             (ph/paragraph [:knowledge-dir-name] (str/join "/" (drop-last (:path file)))))))

(defn- get-aspect-row [aspect files]
  (let [number_of_files 10
        sorted-files (->> files
                          (sort-by :revs >)
                          (take number_of_files))
        knowledge-risk-style (get-knowledge-risk-style aspect)]
    (ph/row
      (ph/cell [:component-cell]
               (ph/paragraph [:heading2 (first (keys knowledge-risk-style))]
                             (first (vals knowledge-risk-style)))
               [:spacer 1]
               (reports-util/concatv
                 (ph/table [:knowledge-table] [1])
                 [(ph/row
                    (ph/cell [:component-header] "Filename"))]
                 (map get-file-row sorted-files))))))

(defn- component-risk-section [item]
  (let [heading (format "Sub-System: %s" (:component item))
        files-group-by-aspects (->> (:files item)
                                    (group-by get-risk-aspect-for))
        aspects (->>(keys files-group-by-aspects)
                    (filter #(not= % :noaspect)))]
    (when (not-empty aspects)
       [(ph/paragraph [:heading2] heading)
       (reports-util/concatv
         (ph/table [:knowledge-table] [1])
         (map #(get-aspect-row % (% files-group-by-aspects)) aspects))
       (ph/pagebreak)])))

(defn- knowledge-loss-section [data options]
  (when (not-empty data)
    (let [component-loss-section-rows (map #(component-loss-section % options) data)]
      [(ph/paragraph [:heading] "System Mastery By Sub-System")
       (ph/paragraph [] "Note: The top files of each component in descending order by knowledge loss")
       (when (some #(some? %) component-loss-section-rows)
         (reports-util/concatv
           (ph/table [:knowledge-table] [1]) component-loss-section-rows))
       (ph/pagebreak)])))

(defn- knowledge-risk-section [data _options]
  (when (not-empty data)
    (reports-util/concatv
      [(ph/paragraph [:heading] "Knowledge Risk By Sub-System")
       (ph/paragraph [] "Note: The top files with risk tag of each component in descending order by number of revisions")
       [:spacer 1]]
        (apply reports-util/concatv (map #(component-risk-section %) data))
      [(ph/pagebreak)])))

(defn key-personnel-and-knowledge-distribution
  "Generate data for pdf generation of key personnel and knowledge distribution report as specified by clj-pdf"
  [data options]
  (concat
    [(with-optional-metadata pdf-reports-def/pdf-report-metadata options)]
    (title-page (update options :title (partial str "[DEPRECATED]\n")))
    (key-personnel-section (:key-personnel data) options)
    (knowledge-loss-section (:knowledge-data-by-component data) options)
    (knowledge-risk-section (:knowledge-data-by-component data) options)))

(defn- max-commits-last-month [report-datas-by-project]
  (->> (vals report-datas-by-project)
       (map :system-health)
       (map :commits-last-month)
       (filter some?)
       (sort >)
       first))

(defn- sort-by-commits-last-month [report-datas-by-project]
  (let [accessor (fn [key]
                   (let [report-data (get report-datas-by-project key)
                         system-health (:system-health report-data)
                         commits-last-month (:commits-last-month system-health)]
                     [commits-last-month key]))]
    (into (sorted-map-by (fn [key1 key2] (compare (accessor key2) (accessor key1))))
          report-datas-by-project)))

(defn management-report-data
  "Generate data for pdf generation as specified by clj-pdf"
  [report-datas-by-project options]
  (let [max-commits-last-month (max-commits-last-month report-datas-by-project)
        report-datas-with-last-values (pick-last-values report-datas-by-project :month)
        sorted-report-datas (sort-by-commits-last-month report-datas-with-last-values)]
    (concat
      [(with-optional-metadata pdf-reports-def/pdf-report-metadata options)]
      (title-page options)
      (mapcat
        (fn [[project-name report-data]]
          (project-summary-row project-name
                               max-commits-last-month
                               (:system-health report-data)
                               (:code-health-overall report-data)
                               (:analysis-dashboard-url report-data)
                               options))
        sorted-report-datas))))

(s/fdef management-report
  :args (s/cat :report-datas-by-project ::reports-spec/report-datas-by-project
               :options ::reports-spec/report-options
               :out some?))
(defn management-report
  "Takes a map of maps of data by name, a map of options, and produces a pdf-report.
  `out` can be either a file name or an output stream"
  [report-datas-by-project options out]
  (pc/pdf (management-report-data report-datas-by-project options) out))

