(ns codescene.features.reports.four-factor-reports
  (:require
   [clojure.java.io :as io]
   [clj-pdf.core :as pc]
   [codescene.features.components.navigation :as navigation]
   [codescene.features.reports.four-factor-pdf :as pdf]
   [codescene.features.reports.common.project :as common-project]
   [codescene.features.reports.portfolio.projects :as portfolio-projects]
   [codescene.features.reports.code-health.project :as code-health-project]
   [tick.core :as t]))

;;; TODO: font path should be passed in as params (in `system`?) so that
;;; we don't have to ensure that Cloud and On-prem use the same resource paths.
(def report-options
  {:font "public/vendor/matter-sq/MatterSQ-Regular.ttf"
   :img-dir "public/imgs/reports"})

(defn- code-health-report-links [system {:keys [id analysis-id]}]
  (merge (common-project/generate-project-links system id analysis-id)
         {:code-red-url "https://codescene.com/hubfs/web_docs/Business-impact-of-code-quality.pdf"}))

(defn- portfolio-report-links [system]
  (let [navigation-component (:navigation-component (:codescene.features.reports/report-component system))
        http-base (navigation/get-http-base navigation-component)
        projects-uri (navigation/get-projects-uri navigation-component)
        four-factor-docs-uri (navigation/get-four-factor-docs-uri navigation-component)
        delivery-config-docs-uri (navigation/get-delivery-config-docs-uri navigation-component)
        global-teams-conf-uri (navigation/get-global-conf-teams-uri navigation-component)
        upgrade-plan-url (navigation/get-upgrade-plan-url navigation-component)]
    {:http-base http-base
     :projects-uri projects-uri
     :four-factor-docs-url (str http-base four-factor-docs-uri)
     :delivery-config-docs-url (str http-base delivery-config-docs-uri)
     :global-teams-conf-url (str http-base global-teams-conf-uri)
     :upgrade-plan-url upgrade-plan-url}))


(defn- file-report-spec
  [path-fn file-name]
  (when path-fn
    (let [relative-report-dir (str "reports/" (random-uuid))
          absolute-report-dir (path-fn relative-report-dir)]
      {:report-dir absolute-report-dir
       :file-name file-name
       :full-file-name (str absolute-report-dir "/" file-name)
       :content-type "application/pdf"})))


(defn- output-report-file [{:keys [full-file-name] :as file-report-spec} pdf-data]
  (io/make-parents full-file-name)
  (with-open [out (io/output-stream full-file-name)]
    (pc/pdf pdf-data out))
  file-report-spec)


(defn generate-code-health-report [system project-id]
  (let [date-now (t/now)
        project {:id project-id :name (common-project/project-name system project-id)}
        {:keys [path-fn name] :as project-with-analysis-info} (common-project/with-analysis-info system project)
        file-report-spec (assoc
                          (file-report-spec path-fn "code-health-report.pdf")
                           :name name
                           :report-generated-on date-now)
        report-data (code-health-project/with-results project-with-analysis-info)
        options (merge report-options (code-health-report-links system project-with-analysis-info))
        pdf-data (pdf/code-health-report report-data options date-now)]
    (if (:analysis-id project-with-analysis-info)
      (output-report-file file-report-spec pdf-data)
      (throw (ex-info (format "Cannot generate the Code Health report for project: %s when there are no analyses yet.
        Run the analysis for the project at least once." project-id) {:project-id project-id})))))


(defn generate-portfolio-report
  [system projects account-info]
  (let [date-now (t/now)
        projects-with-path-fn (portfolio-projects/with-analysis-infos-and-links system projects)
        file-report-spec (assoc
                          (file-report-spec (some :path-fn projects-with-path-fn) "portfolio-overview-report.pdf")
                           :report-generated-on date-now)
        projects-with-results (portfolio-projects/with-results projects-with-path-fn "month")
        data (assoc projects-with-results :account-info account-info)
        options (merge report-options (portfolio-report-links system))
        pdf-data (pdf/portfolio-report data options date-now)]
    (cond
      (empty? projects)
      (throw (ex-info "Cannot generate the Portfolio report as the requester has no accessible projects." {}))
      (empty? projects-with-path-fn)
      (throw (ex-info "Cannot generate the Portfolio report as the accessible projects have no analyses yet.
        Run the analysis for your projects at least once." {}))
      :else
      (output-report-file file-report-spec pdf-data))))


(comment
 (let [options (merge report-options
                      {:http-base "http://www.some.thing/"
                       :dashboard-url "http://www.some.thing/cake/"
                       :pr-conf-url "http://www.some.thing/cookie/"
                       :alerts-uri-fn (partial str "bake/")
                       :code-red-url "https://codescene.com/hubfs/web_docs/Business-impact-of-code-quality.pdf"})
       path-fn (partial codescene.analysis.paths/make-analysis-path-to
                        #_"/home/aasa/Downloads/analysis20230524024601"
                        "/home/ulrikawiss/Downloads/codescene-staging-220622/results(3)/analysis20230622024601"
                        #_"/home/ulrikawiss/Downloads/zookeeper-printegration/results(7)/analysis20230723122501")
       rd (code-health-project/with-results {:id 123 :analysis-id 234 :path-fn path-fn :name "The >< project"})
       pdf-data (pdf/code-health-report rd options)]
   (output-report-file {:full-file-name "reports/chnew2.pdf"} pdf-data)))


(comment
 (let [projects (clojure.edn/read-string {:readers *data-readers*}
                                         (slurp (io/resource "codescene/features/reports/products-onprem.edn")))
       path-fn1 (partial codescene.analysis.paths/make-analysis-path-to
                         #_"/home/ulrikawiss/Downloads/ch-bugs/40054/results(1)/analysis20230531010101"
                         #_"/home/ulrikawiss/Downloads/churntrendinvestig/results (4)/analysis20230508024601"
                         "/home/ulrikawiss/Downloads/codescene-staging-220622/results(3)/analysis20230622024601"
                         #_"/home/ulrikawiss/Downloads/zookeeper-portfolioreport/results(2)/analysis20230621122501"
                         #_"/home/ulrikawiss/Downloads/ZZZ-OBSOLETE_empear-enterprise/results(1)/analysis20210921150102")
       path-fn2 (partial codescene.analysis.paths/make-analysis-path-to
                         "/home/ulrikawiss/Downloads/churntrendinvestig/results (4)/analysis20230508024601"
                         #_"/home/ulrikawiss/Downloads/ZZZ-OBSOLETE_empear-enterprise/results(1)/analysis20210921150102"
                         #_"/home/ulrikawiss/Downloads/zookeeper-portfolioreport/results(2)/analysis20230621122501")
       mock-project-links {:http-base "http://www.some.thing/"
                           :dashboard-url "http://www.some.thing/dashboard/"
                           :pr-stats-url "http://www.some.thing/pr-stats/"
                           :pr-conf-url "http://www.some.thing/pr/"
                           :pm-conf-url "http://www.some.thing/pm/"
                           :teams-conf-url "http://www.some.thing/teams/"
                           :alerts-uri-fn (partial str "alerts/")}
       projects-with-path-fn-and-links (->> projects
                                            (map #(merge %1 %2) [{:id 123 :analysis-id 234 :path-fn path-fn1} {:id 123 :analysis-id 234 :path-fn path-fn2}])
                                            (map #(assoc % :links mock-project-links)))
       ten-projects (concat projects-with-path-fn-and-links
                            projects-with-path-fn-and-links
                            projects-with-path-fn-and-links
                            projects-with-path-fn-and-links
                            projects-with-path-fn-and-links)

       report-data (assoc (portfolio-projects/with-results
                           (concat ten-projects ten-projects ten-projects ten-projects ten-projects
                                   ten-projects ten-projects ten-projects ten-projects ten-projects)
                           #_(concat projects-with-path-fn-and-links
                                   projects-with-path-fn-and-links
                                   projects-with-path-fn-and-links
                                   projects-with-path-fn-and-links
                                   projects-with-path-fn-and-links)
                           #_projects-with-path-fn-and-links
                           "month")
                     :account-info {:user-name "hellosdljflds flzx,cnxzc ljzclz clzxjc lxz clxzk jclxzkj clkxzjclkxzjc lkzjf lisajfpids fpids jflidsj flkdj flkdzj flkdzj flkdzj flkdzj flkdj flkdzj flkdj flkdjz fldskjflds jfldskjfldskjf ldskjflkdsjf ldskjf ldskjf ldskjf ldskjfldsk flkds jfldskf lds"
                                    :org-name "Goodie good good"
                                    :active-authors [{:name "a"}]
                                    :delivery-performance? false})
       options (merge report-options {:http-base "http://cake.bake/"
                                      :four-factor-docs-url "https://codescene.io/docs/4f-dashboard/4-factors-dashboard.html"
                                      :delivery-config-docs-url "delivery-config-docs-url"
                                      :global-teams-conf-url "global-teams-conf-url"
                                      :upgrade-plan-url "upgrade-plan-url"})
       pdf-data (pdf/portfolio-report report-data options)]
   (def proj (:projects report-data))
   (def p (:renderable-trends (first (:projects report-data))))
   (output-report-file {:full-file-name "reports/pf3.pdf"} pdf-data)))


