(ns codescene.features.pm-data.github.github-ticket-id-extractor
  "This namespace contains fns handling GitHub Issue id extraction and generic ticket id extraction 
   from GitHub PRs."
  (:require [codescene.features.pm-data.extractor-common :as ec]
            [codescene.features.pm-data.github.github-fetcher :as f]
            [codescene.features.pm-data.github.github-cache :as gc]
            [codescene.pm-data.provider-common :as c]
            [codescene.url.url-utils :as url]))

;; match issue nbrs with an optional scope,ie '#123' as well as 'owner/repo#123' 
(def ticket-id-pattern #"([\w-]+\/[\w-]+)?#(\d+)")

(defn scoped-ticket-id [owner repo number]
  (format "%s/%s#%s" owner repo number))

(defn- fetch-pull-request-numbers [{:keys [api-client] :as _provider-def} repo-url]
  (if-let [{:keys [owner repo]} (url/github-repo-url->parts repo-url)]
    (->> (gc/fetch-pull-request-numbers nil api-client owner repo)
         (map (fn [{:keys [id]}] (scoped-ticket-id owner repo id))))
    []))

(defn- ->scoped-ticket-id
  "Uses the capture groups from the match to decide on how to construct a scoped ticket id"
  [owner repo [match scope id]]
  (if scope
    match
    (scoped-ticket-id owner repo id)))

(defn extract-ticket-ids [{:keys [owner repo] :as _repo-parts} msg]
  (->> msg
       (re-seq ticket-id-pattern)
       (map #(->scoped-ticket-id owner repo %))))

(defn make-pr-lookup [provider-def {:keys [repo-urls] :as _pm-data-context}]
  (->> repo-urls
       (mapcat #(fetch-pull-request-numbers provider-def %))
       (into #{})))

(defn make-pr-ticket-ids-lookup [provider-def repo-parts]
  (ec/make-pr-ticket-ids-lookup provider-def (partial extract-ticket-ids repo-parts)))

(defn make-ticket-id-extractor [provider-def repo-url pm-data-context]
  (let [repo-parts (url/github-repo-url->parts repo-url)
        pr-lookup (make-pr-lookup provider-def pm-data-context)
        ->pr-ticket-ids (make-pr-ticket-ids-lookup provider-def repo-parts)]
    (fn [rev _date-time msg]
      (when repo-parts
        (->> (concat (extract-ticket-ids repo-parts msg)
                     (->pr-ticket-ids (subs rev 0 7))) ;; TODO: Use entire hash
             (remove pr-lookup))))))

(defn validate-settings
  [{:keys [api-client] :as _provider-def}
   {:keys [repo-urls] :as _pm-data-context}]
  (doseq [{:keys [owner repo]} (map url/github-repo-url->parts repo-urls)]
    (f/fetch-pull-request-count api-client owner repo)))

(comment
  (def repo-url "https://github.com/dakrone/clj-http.git")
  (def repo-url "git@github.com:empear-analytics/analysis-target.git")
  (let [api-client (System/getenv "GITHUB_TOKEN")
        pr-data-provider (codescene.features.pm-data.github.pr-data-provider/->GitHubProvider
                          {:api-client api-client}
                          {:repo-urls [repo-url]})
        provider-def {:map-commits-to-pull-request-issues "on"
                      :api-client api-client
                      :pr-data-provider pr-data-provider}
        repo-parts (url/github-repo-url->parts repo-url)]
    (validate-settings provider-def {:repo-urls [repo-url]})
    (make-pr-ticket-ids-lookup provider-def repo-parts))
  )
