(ns codescene.features.components.navigation
  (:require [clojure.spec.alpha :as s]
            [codescene.features.spec.commons :as commons-spec]
            [codescene.features.components.core :refer [find-component]]))

(defprotocol NavigationComponent
  (-get-http-base [this])
  (-get-projects-uri [this])
  (-get-analysis-results-uri [this project-id analysis-id])
  (-get-knowledge-loss-uri [this project-id analysis-id])
  (-get-pr-statistics-uri [this project-id analysis-id])
  (-get-warning-uri [this project-id analysis-id warning])
  (-get-project-conf-qg [this project-id])
  (-get-project-conf-pr-integration-uri [this project-id])
  (-get-project-conf-teams-uri [this project-id])
  (-get-project-conf-pm-integration-uri [this project-id])
  (-get-global-conf-teams-uri [this])
  (-get-four-factor-docs-uri [this])
  (-get-delivery-config-docs-uri [this])
  (-get-upgrade-plan-url [this]))

(s/fdef get-projects-uri
  :args (s/cat :this some?)
  :ret string?)
(defn get-projects-uri
  "URI to the projects overview page. In OnPrem this can be http-base since there's no redirects,
  but in cloud it will have /projects attached so that our redirects don't send users to codescene.com"
  [this]
  (-get-projects-uri this))

(s/fdef get-http-base
  :args (s/cat :this some?)
  :ret string?)
(defn get-http-base
  "The base URI for the service, including any view contexts, i.e. router base"
  [this]
  (-get-http-base this))

(s/fdef get-analysis-results-uri
  :args (s/cat :this some?
               :project-id ::commons-spec/id
               :analysis-id ::commons-spec/id)
  :ret string?)
(defn get-analysis-results-uri
  "Returns the URI that points to the analysis results, i.e. the analysis dashboard"
  [this project-id analysis-id]
  (-get-analysis-results-uri this project-id analysis-id))

(s/fdef get-knowledge-loss-uri
  :args (s/cat :this some?
               :project-id ::commons-spec/id
               :analysis-id ::commons-spec/id)
  :ret string?)
(defn get-knowledge-loss-uri
  "Returns knowledge loss url for given project-id and analysis-id"
  [this project-id analysis-id]
  (-get-knowledge-loss-uri this project-id analysis-id))

(s/fdef get-pr-statistics-uri
  :args (s/cat :this some?
               :project-id ::commons-spec/id
               :analysis-id ::commons-spec/id)
  :ret string?)
(defn get-pr-statistics-uri
  "Returns PR statistics uri for given project-id and analysis-id"
  [this project-id analysis-id]
  (-get-pr-statistics-uri this project-id analysis-id))

(s/fdef get-warning-uri
  :args (s/cat :this some?
               :project-id ::commons-spec/id
               :analysis-id ::commons-spec/id
               :warning string?)
  :ret string?)
(defn get-warning-uri
  "Returns warning uri for given project-id and analysis-id and warning"
  [this project-id analysis-id warning]
  (-get-warning-uri this project-id analysis-id warning))

(s/fdef get-project-conf-pr-integration-uri
  :args (s/cat :this some?
               :project-id ::commons-spec/id)
  :ret string?)
(defn get-project-conf-pr-integration-uri
  "Returns project configuration / PR integration uri for given project-id"
  [this project-id]
  (-get-project-conf-pr-integration-uri this project-id))

(s/fdef get-project-conf-teams-uri
  :args (s/cat :this some?
               :project-id ::commons-spec/id)
  :ret string?)
(defn get-project-conf-teams-uri
  "Returns project configuration / teams uri for given project-id"
  [this project-id]
  (-get-project-conf-teams-uri this project-id))

(s/fdef get-project-conf-pm-integration-uri
  :args (s/cat :this some?
               :project-id ::commons-spec/id)
  :ret string?)
(defn get-project-conf-pm-integration-uri
  "Returns project configuration / PM integration uri for given project-id"
  [this project-id]
  (-get-project-conf-pm-integration-uri this project-id))

(s/fdef get-global-conf-teams-uri
  :args (s/cat :this some?)
  :ret string?)
(defn get-global-conf-teams-uri
  "Global configuration uri for team configuration"
  [this]
  (-get-global-conf-teams-uri this))

(s/fdef get-four-factor-docs-uri
  :args (s/cat :this some?)
  :ret string?)
(defn get-four-factor-docs-uri
  "Docs page for four factors"
  [this]
  (-get-four-factor-docs-uri this))

(s/fdef get-delivery-config-docs-uri
  :args (s/cat :this some?)
  :ret string?)
(defn get-delivery-config-docs-uri
  "Returns docs page for delivery configuration"
  [this]
  (-get-delivery-config-docs-uri this))

(s/fdef get-upgrade-plan-url
  :args (s/cat :this some?)
  :ret string?)
(defn get-upgrade-plan-url
  "Returns URI for plan upgrade"
  [this]
  (-get-upgrade-plan-url this))

(s/fdef get-project-conf-qg
        :args (s/cat :this some?
                     :project-id ::commons-spec/id)
        :ret string?)
(defn get-project-conf-qg
  "Returns URI for Quality Gates config for a project"
  [this project-id]
  (-get-project-conf-qg this project-id))


(defn component [system] (find-component system :codescene.features.components/navigation))
