(ns codescene.features.code-coverage.parsers.open-cover-parser
  (:require [codescene.features.code-coverage.parser :refer [Parser]]
            [codescene.features.code-coverage.parsers.utils :as utils]
            [codescene.features.code-coverage.parsers.xml-utils :as xml-utils]
            [clojure.java.io :as io]
            [clojure.data.xml :as xml]
            [clojure.string :as str]
            [medley.core :as m]))

(def ^:private supported-metrics [:line-coverage
                                  :sequence-point-coverage
                                  :method-coverage 
                                  :branch-coverage])

(defn- covered-seq-point? [{:keys [attrs] :as _line-el}]
  (some-> attrs :vc Integer/parseInt pos?))

(defn- uncovered-seq-point? [{:keys [attrs] :as _line-el}]
  (some-> attrs :vc Integer/parseInt zero?))

(defn- line-num [{:keys [attrs] :as _line-el}]
  (:sl attrs))

(defn- branch-coverage
  [{:keys [numBranchPoints visitedBranchPoints] :as _summary}]
  (utils/->entry visitedBranchPoints numBranchPoints))

(defn- method-coverage
  [{:keys [numMethods visitedMethods] :as _summary}]
  (utils/->entry visitedMethods numMethods))

(defn- sequence-point-coverage
  [{:keys [numSequencePoints visitedSequencePoints] :as _summary}]
  (utils/->entry visitedSequencePoints numSequencePoints))

(defn- lines-matching 
  [covered-uncovered-fn? seq-points]
  (->> seq-points
       (filter covered-uncovered-fn?) 
       (map line-num)
       set))

(defn- line-coverage
  [parse-options seq-point-els]
  (let [lines (->> seq-point-els (map line-num) distinct count)
        covered   (lines-matching covered-seq-point?   seq-point-els)
        uncovered (lines-matching uncovered-seq-point? seq-point-els)]
    (merge (utils/->entry (count covered) lines)
           (when (utils/needs-detailed-line-coverage? parse-options) 
             {:details {:covered covered 
                        :uncovered uncovered}}))))

(defn- parse-file-el [parse-options ->summary ->seq-points {:keys [attrs] :as _el}]
  (let [uid (:uid attrs)
        path (some-> (:fullPath attrs)
                     (str/replace "\\" "/"))
        summary (->summary uid)
        seq-points (->seq-points uid)]
    (m/assoc-some {:path              path
                   :name              (-> path (str/split #"/") last)}
                 :branch-coverage (branch-coverage summary)
                 :method-coverage (method-coverage summary)
                 :sequence-point-coverage (sequence-point-coverage summary)
                 :line-coverage (line-coverage parse-options seq-points))))

 (def used-summary-key?
   #{:numSequencePoints :visitedSequencePoints
     :numBranchPoints :visitedBranchPoints
     :numMethods :visitedMethods})

(defn- merge-summary-attrs
  [acc method-el]
  (let [file-uid (-> (xml-utils/sub-node :FileRef method-el) :attrs :uid)
        attrs (->> (xml-utils/sub-node :Summary method-el)
                   :attrs
                   (filter (fn [[k _v]] (used-summary-key? k))) ;; Keep only used keys
                   (#(for [[k v] %] [k (Integer/parseInt v)]))  ;; parse values as ints
                   (into {}))]
    (update acc file-uid #(merge-with + % attrs))))

(defn- ->summary-lookup [method-els]
  (reduce merge-summary-attrs {} method-els))

(defn- ->seq-points-lookup [seq-point-els]
  (group-by #(-> % :attrs :fileid) seq-point-els))
 
(defn- read-coverage* [reader parse-options]
  (let [xml (-> reader (xml/parse :namespace-aware false :support-dtd false))
        files (xml-utils/sub-nodes-in xml [:Modules :Module :Files :File])
        methods (xml-utils/sub-nodes-rec :Method xml)
        seq-points (xml-utils/sub-nodes-rec :SequencePoint xml)
        ->seq-points (->seq-points-lookup seq-points)
        ->summary (->summary-lookup methods)]
    (map (partial parse-file-el parse-options ->summary ->seq-points) files)))

(defn- read-coverage [f parse-options]
  (with-open [r (io/reader f)]
    (doall (read-coverage* r parse-options))))

(defn ->Parser []
  (reify Parser
    (-read-coverage [this reader parse-options] (read-coverage* reader parse-options))
    (-supported-metrics [this] supported-metrics)
    (-id [this] "open-cover")
    (-name [this] "OpenCover")))

(comment
  (def f "./test/codescene/features/code_coverage/testdata/CSharp/OpenCover_altcover.xml")
  (def parse-options {:cli-command "check"})
  (read-coverage f parse-options)
  )
