(ns codescene.features.code-coverage.parsers.open-clover-parser
  (:require [codescene.features.code-coverage.parser :refer [Parser]]
            [codescene.features.code-coverage.parsers.xml-utils :as xml-utils]
            [codescene.features.code-coverage.parsers.utils :as utils]
            [clojure.java.io :as io]
            [clojure.data.xml :as xml]
            [clojure.string :as str]
            [medley.core :as m]))

(def ^:private supported-metrics [:line-coverage
                                  :method-coverage
                                  :condition-coverage
                                  :statement-coverage])

(defn- covered-line? [{:keys [attrs] :as _line-el}]
  (let [{:keys [count]} attrs]
    (some-> count Integer/parseInt pos?)))

(defn- uncovered-line? [{:keys [attrs] :as _line-el}]
  (let [{:keys [count]} attrs]
    (some-> count Integer/parseInt zero?)))

(defn- line-num [{:keys [attrs] :as _line-el}]
  (:num attrs))

(defn- lines-matching 
  [covered-uncovered-fn? all-lines]
  (->> all-lines 
       (filter covered-uncovered-fn?)
       (map line-num)
       set))

(defn attrs->coverage [{:keys [attrs] :as _el} covered-attr total-attr]
  (let [covered-str (get attrs covered-attr)
        total-str (get attrs total-attr)]
    (when (and covered-str total-str)
      (utils/->entry (Integer/parseInt covered-str) (Integer/parseInt total-str)))))

(defn- method-coverage [metrics-el]
  (attrs->coverage metrics-el :coveredmethods :methods))

(defn- condition-coverage [metrics-el]
  (attrs->coverage metrics-el :coveredconditionals :conditionals))

(defn- statement-coverage [metrics-el]
  (attrs->coverage metrics-el :coveredstatements :statements))

(defn- line-coverage
  [parse-options
   line-els]
  (let [lines (->> line-els (map line-num) distinct count)
        covered   (lines-matching covered-line? line-els)
        uncovered (lines-matching uncovered-line? line-els)
        n-covered (count covered)]
    (when (pos? lines)
      (merge
       {:covered n-covered
        :total lines
        :coverage (/ n-covered (max 1 lines))}
       (when (utils/needs-detailed-line-coverage? parse-options)
         {:details {:covered   covered
                    :uncovered uncovered}})))))

(defn parse-file-el 
  [parse-options
   {:keys [attrs] :as el}]
  (let [path (-> (:path attrs)
                 (str/replace "\\" "/"))
        name (:name attrs)
        metrics (xml-utils/sub-node :metrics el)
        lines (xml-utils/sub-nodes :line el)]
    ;; See also http://openclover.org/doc/manual/4.2.0/faq--how-are-coverage-percentages-calculated.html
    (m/assoc-some {:path path
                   :name name}
                  :method-coverage (method-coverage metrics)
                  :condition-coverage (condition-coverage metrics)
                  :statement-coverage (statement-coverage metrics)
                  :line-coverage (line-coverage parse-options lines))))

(defn- read-coverage* 
  [reader
   parse-options]
  (let [xml (-> reader (xml/parse :namespace-aware false :support-dtd false))
        file-els (xml-utils/sub-nodes-in xml [:project :package :file])
        test-file-els (xml-utils/sub-nodes-in xml [:test-project :package :file])]
    (map (partial parse-file-el parse-options) (concat file-els test-file-els))))

(defn- read-coverage [f parse-options]
  (with-open [r (io/reader f)]
    (doall (read-coverage* parse-options r))))

(defn ->Parser []
  (reify Parser
    (-read-coverage [this reader parse-options] (read-coverage* reader parse-options))
    (-supported-metrics [this] supported-metrics)
    (-id [this] "open-clover")
    (-name [this] "OpenClover")))

(comment
  (def f "./test/codescene/features/code_coverage/testdata/clover.xml")
  (def parse-options {:cli-command "check"})
  (read-coverage f parse-options)
  )