(ns codescene.features.api.routes.projects
  (:require [compojure.api.sweet :refer [routes GET POST PUT DELETE context]]
            [spec-tools.spec :as spec]
            [spec-tools.data-spec :as ds]
            [codescene.features.components.project :as project-component]
            [codescene.features.components.auth :as auth-component]
            [clojure.spec.alpha :as s]
            [codescene.features.api.privileges :as api-privileges]
            [codescene.features.api.core :as api-core]
            [codescene.features.api.spec.analysis :as api-analysis-spec]
            [codescene.features.api.spec.project :as api-project-spec]
            [codescene.features.api.spec.badge :as api-badges-spec]
            [codescene.features.api.projects :as projects]))

(defn- repository-sub-routes
  [system project-id]
  (routes
   (POST "/repository" []
     :middleware [#(api-core/wrap-authorize-project system % project-id #{api-privileges/restapi-write})]
     :summary "Add repository to project"
     :tags ["projects"]
     :body [data (ds/spec {:spec (s/keys :req-un [::api-analysis-spec/repositories] :opt-un [::api-analysis-spec/generate-architectural-component]) :name :repository})]
     :responses {200 {:schema      spec/any?
                      :description (str "add repositories to project, by default an architectural component will be generated as well, "
                                        "you can overwrite this behaviour using \"generate-architectural-component\": true/false parameter")}}
     (projects/add-repositories-result system project-id data))
   (PUT "/repository" []
     :middleware [#(api-core/wrap-authorize-project system % project-id #{api-privileges/restapi-write})]
     :summary "Edit repository branch on project"
     :tags ["projects"]
     :body [data (ds/spec {:spec (s/keys :req-un [::api-analysis-spec/repositories]) :name :repositories})]
     :responses {200 {:schema      spec/any?
                      :description "edit repositories branch to project"}}
     (projects/edit-repositories-result system project-id data))
   (DELETE "/repository/remove" []
     :middleware [#(api-core/wrap-authorize-project system % project-id #{api-privileges/restapi-write})]
     :summary "Remove repository URL's from project"
     :tags ["projects"]
     :body [data (ds/spec {:spec (s/keys :req-un [::api-analysis-spec/urls]) :name :repositories})]
     :responses {200 {:schema      spec/any?
                      :description "remove repository urls from project"}}
     (projects/remove-repositories-result system project-id data))))

(defn sub-routes
  [system]
  (routes
   (GET "/projects" request
     :summary "Project Lists"
     :tags ["projects"]
     :query-params [{page :- spec/int? 1}
                    {order_by :- ::api-project-spec/project_list_order_by ""}
                    {fields :- spec/string? ""}
                    {filter :- spec/string? ""}]
     :responses {200 {:schema      ::api-analysis-spec/project_list
                      :description "project list for the authenticated user"}}
     :middleware [#(api-core/wrap-authorize-any system % #{api-privileges/restapi-read api-privileges/mcp-api-read})]
     (projects/list-result system request {:page page
                                           :order-by order_by
                                           :fields fields
                                           :filter filter}))
   (POST "/projects/new" request
     :summary "Import New Project"
     :tags ["projects"]
     :body [data (ds/spec {:spec (s/keys :req-un [::project-component/config]) :name :project-config})]
     :responses {201 {:schema      spec/any?
                      :description "the project json configuration to import"}}
     :middleware [#(api-core/wrap-authorize-any system % #{api-privileges/restapi-write})]
     (projects/post-new-project-result system request data))))

(defn- project-by-id-sub-routes-reads
  [system project-id]
  (context "/" [] ;; Read routes
    :middleware [#(api-core/wrap-authorize-project system % project-id #{api-privileges/restapi-read})]
    (GET "/" []
      :summary "Get Project by id"
      :tags ["projects"]
      :responses {200 {:schema ::api-analysis-spec/project
                       :description "project by id"}}
      (projects/project-result system project-id))
    (GET "/badges" []
      :summary "Get Project Badges Status"
      :tags ["projects"]
      :responses {200 {:schema ::api-badges-spec/badges
                       :description "badge status for project id"}}
      (projects/badges-result system project-id))
    (GET "/repositories" []
      :middleware [#(api-core/wrap-authorize-project system % project-id #{api-privileges/restapi-read})]
      :summary "Get repositories of the project"
      :tags ["projects"]
      :responses {200 {:schema      ::api-analysis-spec/repositories
                       :description "repositories list for the project id"}}
      (projects/all-repositories-result system project-id))
    (GET "/components" []
      :middleware [#(api-core/wrap-authorize-project system % project-id #{api-privileges/restapi-read})]
      :summary "Get architectural components config of the project"
      :tags ["projects"]
      :responses {200 {:schema      ::api-project-spec/component-config-list
                       :description "architectural components config for the project id"}}
      (projects/components-config-result system project-id))))

(defn- project-by-id-sub-routes-writes
  [system project-id]
  (context "/" [] ;; Write routes
    (PUT "/badges" []
      :middleware [#(api-core/wrap-authorize-project system % project-id #{api-privileges/restapi-write})]
      :summary "Update Project Badge Status"
      :tags ["projects"]
      :body [data (ds/spec {:spec api-badges-spec/badges-spec :name :badge-data})]
      :responses {200 {:schema      spec/any?
                       :description "project badgse status update. allowed values are \"code-health\", \"missed-goals\" and \"system-mastery\""}}
      (projects/update-badges-result system project-id data))
    (POST "/run-analysis" request
      :middleware [#(api-core/wrap-authorize-project system % project-id #{api-privileges/restapi-write api-privileges/cli-access})]
      :summary "Run an analysis of the project"
      :tags ["projects"]
      :responses {200 {:schema      spec/any?
                       :description "run an analysis of the project"}}
      (projects/run-analysis system request project-id))
    (DELETE "/" request
      :middleware [#(api-core/wrap-authorize-project system % project-id #{api-privileges/restapi-write})]
      :summary "Delete project"
      :tags ["projects"]
      :query-params [{developer-settings :- spec/boolean? true}]
      :responses {200 {:schema      spec/any?
                       :description (str "delete project and all related configurations, all project analyses result are deleted, "
                                         "project developer-settings are removed if not shared with other project even if query param developer-settings=true")}}
      (let [auth-component (api-core/api-auth system)
            user (auth-component/user auth-component request)]
        (projects/delete-project system user project-id developer-settings)))
    (POST "/components" []
      :middleware [#(api-core/wrap-authorize-project system % project-id #{api-privileges/restapi-write})]
      :summary "Update architectural components config of the project"
      :tags ["projects"]
      :body [data (ds/spec {:spec (s/coll-of ::api-project-spec/component-config) :name :components-config})]
      :responses {201 {:schema      spec/any?
                       :description "the result of the architectural components config update"}}
      (projects/post-components-config-result system project-id data))
    (repository-sub-routes system project-id)))

(defn project-by-id-sub-routes
  [system project_id]
  (routes
   (project-by-id-sub-routes-writes system project_id)
   (project-by-id-sub-routes-reads system project_id)))
