(ns codescene.features.api.projects
  (:require [camel-snake-kebab.core :as csk]
            [camel-snake-kebab.extras :as cske]
            [clojure.string :as str]
            [codescene.features.analysis.util :as analysis-util]
            [codescene.features.api.analyses :as analyses]
            [codescene.features.api.core :as api-core]
            [codescene.features.components.analysis :as comp.analysis]
            [codescene.features.components.auth :as auth]
            [codescene.features.components.project :as project]
            [codescene.features.util.api :as api-utils]
            [medley.core :as m]
            [slingshot.slingshot :refer [try+]]
            [taoensso.timbre :as log]))

(defn- with-ref
  [api-prefix project]
  (assoc project :ref (str (api-utils/project-url-prefix api-prefix) (:id project))))

(defn- add-project-analysis-data
  [system {:keys [id] :as project}]
  (let [analysis (api-core/api-analyses system)
        summary (comp.analysis/latest-successful-analysis-summary analysis id)]
    (if summary
      (assoc project :analysis summary)
      project)))

(defn list-result
  [system request {:keys [page order-by fields] filter-str :filter}]
  (let [projects-component (api-core/api-projects system)
        auth-component (api-core/api-auth system)
        user (auth/user auth-component request)
        all-accessible-projects (->> (project/all-accessible-projects projects-component user)
                                     (map #(select-keys % [:id :name]))
                                     (map #(add-project-analysis-data system %)))
        api-prefix (api-core/api-root system)
        add-api-ref (partial with-ref api-prefix)
        transform-keys (partial cske/transform-keys csk/->snake_case_keyword)]
    (-> (map #(-> % add-api-ref transform-keys) all-accessible-projects)
        (analysis-util/paginated-ordered-data {:page page
                                               :page-size api-utils/default-page-limit
                                               :order-by order-by
                                               :fields fields
                                               :filter filter-str
                                               :results-k :projects})
        api-utils/ok)))

(defn post-new-project-result
  [system request data]
  (try+
    (let [projects-component (api-core/api-projects system)
          auth-component (api-core/api-auth system)
          user (auth/user auth-component request)
          project-created (project/create-new-project projects-component user data)
          ref (str (api-utils/project-url-prefix (api-core/api-root system)) (:id project-created))
          message (str "project-created: " (:name project-created))]
      (if project-created
        (api-utils/created {:ok message :ref ref :id (:id project-created)} {"Location" ref})
        (api-utils/bad-request "There was an error while creating the project, please check the logs!!")))
    (catch [] e
      (log/error (:throwable &throw-context) (:message &throw-context))
      (api-utils/bad-request e))))

(defn badges-result
  [system project-id]
  (let [projects-component (api-core/api-projects system)]
    (api-utils/ok (merge {:average-code-health false :hotspot-code-health false :missed-goals false :system-mastery false}
                         (project/badge-status projects-component project-id)))))

(defn update-badges-result
  [system project-id data]
  (let [projects-component (api-core/api-projects system)
        params (->> (into [] data)
                    (filter #(second %))
                    (map #(-> % first name (str/replace "_" "-"))))]
    (project/badge-status-update projects-component project-id params)
    (api-utils/ok {:ok (format "badges for project-id: %s updated" project-id)})))

(defn project-result
  [system project-id]
  (let [projects-component (api-core/api-projects system)
        {:keys [repositories] :as project} (project/project-by-id projects-component project-id)
        configured-git-remote-urls (if (some? repositories)
                                     (map :path repositories)
                                     (->> (project/all-repositories projects-component project-id)
                                         (map :url)))]
    (api-utils/ok (-> (add-project-analysis-data system project)
                      (select-keys [:id :name :description :project_owner :analysis-destination :repository-paths-on-disk :analysis])
                      (assoc :configured_git_remote_urls configured-git-remote-urls)))))

(defn all-repositories-result
  [system project-id]
    (let [projects-component (api-core/api-projects system)]
      (api-utils/ok (project/all-repositories projects-component project-id))))


(defn add-repositories-result
  [system project-id repositories]
  (try+
    (let [projects-component (api-core/api-projects system)
          response (project/add-repositories projects-component project-id repositories)]
      (api-utils/ok {:ok "project-updated" :repo_paths response}))
    (catch [] e
      (api-utils/bad-request e))))

(defn edit-repositories-result
  [system project-id repositories]
  (try+
    (let [projects-component (api-core/api-projects system)
          response (project/edit-repositories projects-component project-id repositories)]
      (api-utils/ok {:ok "project-updated" :repo_paths response}))
    (catch [] e
      (api-utils/bad-request e))))

(defn remove-repositories-result
  [system project-id repositories]
  (try+
    (let [projects-component (api-core/api-projects system)
          response (project/remove-repositories projects-component project-id repositories)]
      (api-utils/ok {:ok "project-updated" :repo_paths response}))
    (catch [] e
      (api-utils/bad-request e))))

(defn run-analysis
  [system request project-id]
  (try+
    (let [projects-component (api-core/api-projects system)
          auth-component (api-core/api-auth system)
          user (auth/user auth-component request)
          {:keys [id html-ref]} (project/run-analysis projects-component user project-id)]
      (api-utils/accepted (m/assoc-some {:ok "analysis-scheduled"}
                                        :id id
                                        :ref (when id (analyses/analysis-ref (api-core/api-root system) project-id id))
                                        :html_ref html-ref)
                         {}))
    (catch [:type :invalid-analyses-request] {:keys [message]}
      (api-utils/bad-request message))))

(defn delete-project
  [system user project-id settings?]
  (try+
    (let [projects-component (api-core/api-projects system)]
      (project/delete-project projects-component user project-id settings?)
      (api-utils/ok {:ok "project deleted"}))
    (catch [] e
      (api-utils/bad-request e))))

(defn components-config-result
  [system project-id]
  (let [project-component (api-core/api-projects system)
        components (project/architectural-components project-component project-id)
        transform-fn (fn [[k v]] {:name k :patterns (map :pattern v)})]
    (api-utils/ok (map transform-fn (group-by :component components)))))

(defn post-components-config-result
  [system project-id data]
  (let [project-component (api-core/api-projects system)]
    (project/update-architectural-components project-component project-id data)
    (api-utils/created {:ok "Architectural components config updated."} {})))