(ns codescene.features.api.delta-analyses
  (:require [clojure.set :as set]
            [codescene.delta.delta-result :as delta-result]
            [codescene.features.delta.code-health.ch-gates :as ch-gates]
            [codescene.features.delta.result-presenter :as result-presenter]
            [codescene.features.util.api :as api-utils]
            [codescene.features.api.core :as api-core]
            [codescene.features.components.delta-analysis :as delta-analysis]
            [medley.core :as m]))

(defn- with-ref
  [api-prefix project-id delta-analysis]
  (assoc delta-analysis :ref (format "%s%s/delta-analyses/%s" (api-utils/project-url-prefix api-prefix) project-id (:id delta-analysis))))

(defn- delta-context-props
  [m]
  (select-keys m [:analysistime :id :project-id :repository :external-review-id]))

(defn list-result
  [system project-id page]
  (let [api-prefix (api-core/api-root system)
        delta-analysis (api-core/api-delta-analyses system)
        all-delta-analyses-in-project (->> (delta-analysis/all-in-project delta-analysis project-id)
                                           (map delta-context-props))]
    ;; we need to return a vector to avoid issue from compojure-api which return reverses LazySeqs
    ;;(see https://github.com/metosin/compojure-api/issues/412)
    (api-utils/paginated-list-result (into [] all-delta-analyses-in-project)
                                     {:page page
                                      :list-kw :delta-analyses
                                      :processing-fn (partial with-ref api-prefix project-id)})))
         
(defn- file-results-for-api
  [result]
  (->> (delta-result/file-scores result)
       (map #(set/rename-keys % {:score :code-health
                                 :old-score :old-code-health}))))

(defn- directives-for-api
  [file-results]
  (let [file-directive-for-api (partial map #(set/rename-keys % {:name :file}))]
    (-> (result-presenter/->directives file-results)
        (update :added file-directive-for-api)
        (update :removed file-directive-for-api))))

(defn- delta-results-for-api
  [suppressions {:keys [delta-branch-head file-results] :as result}]
  (let [directives (directives-for-api file-results)
        enabled-gates (ch-gates/enabled-gates-in file-results)
        failed-gates (->> (ch-gates/failed-gates suppressions file-results)
                          (m/map-keys keyword))
        file-results-for-api (file-results-for-api result)]
    (merge (select-keys result
                        [:authors
                         :base-ref
                         :commits
                         :external-review-id])
           {:directives directives
            :enabled-gates enabled-gates
            :failed-gates failed-gates
            :file-results file-results-for-api
            :head-ref delta-branch-head
            :code-health (delta-result/weighted-score file-results-for-api :code-health :loc)
            :old-code-health (delta-result/weighted-score file-results-for-api :old-code-health :old-loc)})))

(defn- delta-analysis-response
  [{:keys [context result] :as _delta}]
  (merge (delta-context-props context) 
         (when result 
           (delta-results-for-api (:suppressions context) result))))

(defn delta-analysis-result
  [system project-id delta-analysis-id]
  (if-let [delta-analysis-by-id (-> (api-core/api-delta-analyses system)
                                    (delta-analysis/by-id project-id delta-analysis-id))]
    (api-utils/ok (delta-analysis-response delta-analysis-by-id))
    (api-utils/not-found (format "There's no delta analysis matching the ID %s for project with ID %s" delta-analysis-id project-id))))
