(ns codescene.features.reports.for_report
  (:require [clj-pdf.core :as pc]
            [clj-time.format :as tf]
            [clojure.java.io :as io]
            [clojure.data.json :as json]
            [clojure.set :as set]
            [clojure.string :as str]
            [codescene.util.csv-conversions :as csv-conversions]
            [codescene.analysis.analysis-context-builder :as analysis-context-builder]
            [codescene.analysis.analysis-warnings :as analysis-warnings]
            [codescene.analysis.paths :as paths]
            [codescene.biomarkers.architectural-level-hotspots :as architectural-hotspots]
            [codescene.branches.branch-statistics :as branch-statistics]
            [codescene.note-watch.supervised-scores :as supervised-scores]
            [codescene.presentation.display :as display]
            [codescene.presentation.health :as health]
            [codescene.presentation.system-health :as system-health]
            [codescene.features.reports.pdf-reports :as pdf-reports]
            [codescene.features.factors.kpi-time-series :as kpi-time-series]
            [codescene.features.util.number :as number-util]
            [codescene.velocity.development-output :as development-output]
            [evolutionary-metrics.mining.file-patterns :as file-patterns]
            [hotspots-x-ray.recommendations.code-health-interpretations :as chi]
            [hotspots-x-ray.recommendations.code-health.user-configureable-rules :as user-configureable-rules]
            [semantic-csv.core :as sc]))

;; TODO: Put these simple generic functions somewhere else... (or find equivalents)
;; Update: started to extract CSV content to csv-conversions.

(defn read-json [json-file]
  (with-open [reader (io/reader json-file)]
    (json/read reader :key-fn keyword)))

(defn read-optional-json [json-file default]
  (if (.exists (io/as-file json-file))
    (read-json json-file)
    default))

(defn- parse-int [s]
  (Integer/parseInt s))

(defn- parse-nilable-int [s]
  (when (and s (not= s "-"))
    (Integer/parseInt s)))

(defn- parse-nilable-percentage [s]
  (when (and s (not= s "-"))
    (/ (Integer/parseInt s) 100.0)))

(defn- parse-nilable-string [s]
  (when (and s (not= s "-"))
    s))

(defn- parse-autonomy [s]
  (if (some? s)
    (get {"High"   :high-team-autonomy
          "Medium" :medium-team-autonomy
          "Low"    :low-team-autonomy
          "-"      :not-calculated} s)
    :not-calculated))

(defn- parse-positive-int [s]
  (when-let [v (parse-nilable-int s)]
    (when (> v 0)
      v)))

(defn- has-score?
  [s]
  (and s (not= s "-")))

(defn- parse-nilable-float [s]
  (when (and s (not= s "-"))
    (Double/parseDouble s)))

(defn parse-code-health-score
  "Legacy scores are single digit, new ones are floats."
  [s]
  (when (has-score? s)
    (if (re-matches #"\d+" s)
      (parse-positive-int s)
      (parse-nilable-float s))))

(defn- date-as-string [date]
  (tf/unparse (:year-month-day tf/formatters) date))

;;TODO: Copy-pasted fn fron enterprise -> refactor!
(defn- highest-delivery-risk-for
  [system-hotspots-health]
  (->> system-hotspots-health
       (map :deliveryrisk)
       (map parse-nilable-int)
       (filter some?)
       (sort >)
       first))

(defn- notes-for [notes file]
  (->> notes
       (map supervised-scores/file-details-for-note)
       (filter #(str/includes? (:resolved %) file))
       (map :note)))

(defn- file-details-for-note->goal [file-details-for-note]
  {:file         (:relative-name file-details-for-note)
   :goal-details (get-in file-details-for-note [:note :note-text])})

;;; TODO: unused. Remove?
(defn- missed-goals-of-category [category file-details-for-note notes-with-warnings]
  (let [missed-ids (->> notes-with-warnings (map :id) (into #{}))
        note-id #(get-in % [:note :id])]
    (->> file-details-for-note
         (filter #(missed-ids (note-id %)))
         (filter :resolved)
         (filter #(= category (get-in % [:note :category])))
         (map file-details-for-note->goal))))

(defn- note-text-for [notes]
  (str/join "\n" (map :note-text notes)))

(defn- note-category-for [notes]
  (str/join "\n" (map :category notes)))

(defn system-health [dashboard risks-for-notes system-hotspots-health classified-hotspot-stats analysis-time]
  (let [notes-warnings (analysis-warnings/note-specific-warnings-in risks-for-notes)
        hotspots (first (filter #(= (:title %1) "Hotspots") classified-hotspot-stats))]
    {:abandoned-code           (parse-nilable-percentage (:knowledgelosspercentage dashboard))
     :active-authors           (parse-nilable-int (:activeauthors dashboard))
     :commits-last-month       (parse-nilable-int (:commitslastmonth dashboard))
     :former-developers        (parse-nilable-int (:knowledgelossauthors dashboard))
     :code-health              (parse-code-health-score (:biomarkercurrent dashboard))
     :last-month-code-health   (parse-code-health-score (:biomarkermonthback dashboard))
     :last-year-code-health    (parse-code-health-score (:biomarkeryearback dashboard))
     :lines-of-code            (parse-nilable-string (:codesize dashboard))
     :analysis-time            analysis-time
     :missed-goals             (::analysis-warnings/supervision-warnings notes-warnings)
     :total-goals              (::analysis-warnings/total-goals notes-warnings)
     :delivery-risk            (highest-delivery-risk-for system-hotspots-health)
     :defect-ratio-in-hotspots (parse-nilable-percentage (:defectratioinhotspots dashboard))
     :red-hotspot-ratio        (when hotspots (/ (first (:measures hotspots)) 100.0))
     :red-hotspot-effort       (when hotspots (/ (first (:efforts hotspots)) 100.0))
     :key-personnel            (parse-nilable-int (:nkeypersonnel dashboard))
     :key-personnel-ratio      (parse-nilable-percentage (:nkeypersonnelpercentage dashboard))}))

(defn early-warnings [risk-by-commit-early-warnings risks-for-notes]
  ;; TODO:  Go from commit,risk,author,date,repository to title,description,early-warning-details (list)
  [])

(defn- branch-statistics-item->delivery-risk [early-warning]
  {:branch-name    (:name early-warning)
   :lead-time      (parse-nilable-int (:leadmerge early-warning))
   :risk           (parse-nilable-int (:risk early-warning))
   :nbr-of-commits (parse-nilable-int (:commits early-warning))})

(defn has-warning? [branch]
  (or (not= (:authorwarning branch) branch-statistics/no-warning)
      (not= (:ttlwarning branch) branch-statistics/no-warning)))

(defn delivery-risks [branch-statistics]
  (->> branch-statistics
       (filter #(= "true" (:calculated %)))
       (map branch-statistics-item->delivery-risk)))

(defn- revision-trend-item->development-output [item]
  {:date             (:date item)
   :commits-per-week (parse-int (:revisions item))
   :authors-per-week (parse-int (:authors item))})

(defn- authors-per-month-item->active-authors [item]
  {:date              (:date item)
   :authors-per-month (parse-int (:authors item))})

(defn- weekly-velocity [revision-trend]
  (->> revision-trend
       (sc/cast-with {:revisions sc/->int :authors sc/->int})
       development-output/weekly-velocity-from
       (map #(update % :date date-as-string))))

(defn system-trends [revision-trend authors-per-month-trend development-output]
  {:development-output-over-time (map revision-trend-item->development-output revision-trend)
   :active-authors-over-time     (map authors-per-month-item->active-authors authors-per-month-trend)
   ;  {"monthly-velocity-delta":-55,"monthly-mean-authors":2.2,"monthly-authors-delta":-0.2999999999999998,"weekly-velocity-delta":240,"weekly-authors-delta":0}
   :weekly-velocity              (weekly-velocity revision-trend)
   ;; The development-output might be empty, report generation wants nil
   :development-output (when (seq development-output) development-output)})

(defn- code-health-for-a-human->hotspot-detail [detail]
  {:severity    (:indication detail)
   :description (:title detail)})

(defn- biomarkers-details-entry-for [code-biomarkers-details file]
  (->> (:markers code-biomarkers-details)
       (filter #(= (:name %) file))
       first))

(defn- hotspot-details-for [context code-biomarkers-details file]
  (let [entry (biomarkers-details-entry-for code-biomarkers-details file)]
    (if (some? entry)
      (map code-health-for-a-human->hotspot-detail (chi/interpret-for-a-human context entry))
      [])))

(defn- code-bio-marker-score->system-hotspot [context notes code-biomarkers-details code-bio-marker-score]
  (let [file (:name code-bio-marker-score)
        notes-for-file (notes-for notes file)]
    {:file                   file
     :code-health            (parse-code-health-score (:score code-bio-marker-score))
     :last-month-code-health (parse-code-health-score (:last-month code-bio-marker-score))
     :last-year-code-health  (parse-code-health-score (:last-year code-bio-marker-score))
     :hotspot-details        (hotspot-details-for context code-biomarkers-details file)
     :goals                  (note-category-for notes-for-file)
     :comment                (note-text-for notes-for-file)}))

(defn- system-hotspot-health->subsystem-health [code-bio-marker-score]
  {:sub-system             (:name code-bio-marker-score)
   :team-autonomy          (parse-autonomy (:teamautonomy code-bio-marker-score))
   :system-mastery         (parse-nilable-percentage (:mastery code-bio-marker-score))
   :code-health            (parse-code-health-score (:current code-bio-marker-score))
   :last-month-code-health (parse-code-health-score (:month code-bio-marker-score))
   :last-year-code-health  (parse-code-health-score (:year code-bio-marker-score))
   :defects                (parse-nilable-int (:defects code-bio-marker-score))
   :last-defects           (parse-nilable-int (:defectsmonth code-bio-marker-score))
   :delivery-risk          (parse-nilable-int (:deliveryrisk code-bio-marker-score))
   :commits                (parse-nilable-int (:revisions code-bio-marker-score))})

(defn- system-hotspots [context notes code-bio-marker-scores code-bio-marker-details-json]
  (->> code-bio-marker-scores
       (filter #(and (= "true" (:calculated %))
                     (= "1" (:presentable %))))
       (map (partial code-bio-marker-score->system-hotspot context notes code-bio-marker-details-json))))

(defn- subsystem-health [system-hotspots-health]
  (map system-hotspot-health->subsystem-health system-hotspots-health))

(defn- hotspots-for-subsystem [{:keys [analysis-path-fn] :as context} notes component]
  (let [analysis-result-directory (analysis-path-fn "")
        code-bio-marker-scores-csv (architectural-hotspots/score-path analysis-result-directory component)
        code-bio-marker-scores (csv-conversions/csv->maps code-bio-marker-scores-csv)
        code-bio-marker-details-json (architectural-hotspots/biomarkers-detail-path analysis-result-directory component)
        code-bio-marker-details (read-json code-bio-marker-details-json)]
    (system-hotspots context notes code-bio-marker-scores code-bio-marker-details)))


(defn- hotspots-by-subsystem [analysis-results-path-fn notes system-hotspots-health]
  (let [components (map :name system-hotspots-health)]
    (->> components
         (map (fn [c] [c (hotspots-for-subsystem analysis-results-path-fn notes c)]))
         (into {}))))

(defn- has-customized-code-health-rules?
  [path-fn]
  (seq (user-configureable-rules/read-human-presentable-from path-fn)))

(defn- code-health-overall
  [path-fn]
  (let [code-health-series (-> (read-optional-json (path-fn paths/code-health-time-series-json) [])
                               kpi-time-series/select-data-points-in-scope)
        transform-fn (fn [m sufix] (update m :key (fn [v] (str v sufix))))
        code-health-now (->> code-health-series last :sub-kpis (map #(transform-fn % "-now")))
        code-health-last-month (->> code-health-series first :sub-kpis (map #(transform-fn % "-last-month")))
        result (->> (concat code-health-now code-health-last-month)
                    (map #(hash-map (-> % :key keyword) (-> % :value number-util/floor-number-when-exists)))
                    (into {}))]
    (set/rename-keys result {:hotspots-code-health-now        :biomarkercurrent
                             :hotspots-code-health-last-month :biomarkermonthback
                             :average-code-health-now         :full-code-health-average-now
                             :average-code-health-last-month  :full-code-health-average-last-month
                             :worst-performer-now             :full-code-health-worst-performer-now
                             :worst-performer-last-month      :full-code-health-worst-performer-last-month})))

(defn- technical-health-and-trend-data [report-spec]
  (let [{:keys [path-fn]} report-spec
        context (analysis-context-builder/build-context-for-existing-analysis path-fn)
        code-bio-marker-scores (csv-conversions/csv->maps (path-fn paths/code-bio-marker-scores-csv))
        code-biomarkers-details (read-json (path-fn paths/code-bio-marker-details-json))
        notes (->> (read-optional-json (path-fn paths/risks-for-notes-json) [])
                   (map :note-score)
                   (map :note))
        system-hotspots-health (-> (path-fn paths/system-hotspots-health-csv)
                                   csv-conversions/optional-csv->maps
                                   (system-health/csv-rows->system-health  "year"))
        early-warnings (:early-warnings report-spec)]
    {:subsystem-health      system-hotspots-health
     :system-hotspots       (system-hotspots context notes code-bio-marker-scores code-biomarkers-details)
     :hotspots-by-subsystem (hotspots-by-subsystem context notes system-hotspots-health)
     :code-health-overall (code-health-overall path-fn)
     :early-warnings early-warnings
     :customized-code-health-rules (has-customized-code-health-rules? path-fn)}))

(defn- flatten-tree
  "flat recursively all :children from the given data"
  [data]
  (if-let [children (:children data)]
    (flatten (map flatten-tree children))
    data))

(defn- get-social-system-data
  [path-fn]
  (let [system-map-json (read-optional-json (path-fn paths/social-system-map-file-name) [])]
    (when (:children system-map-json)
      (-> system-map-json
          flatten-tree))))

(defn- get-files-group-by-component
  [components social-system-map]
  (let [filter-fn (fn [c] (filter #(= c (:component %)) social-system-map))]
      (map #(assoc % :files (filter-fn (:component %))) components)))

(defn- get-component [file components]
  (let [path (str/join "/" (:path file))]
    (-> (filter #((:matcher %) path) components)
        first
        (select-keys [:component :component-mastery]))))

(defn- get-component-mastery [component architectural-knowledge-loss]
  (-> (filter #(= component (:entity %)) architectural-knowledge-loss)
      first
      :ownership
      display/->maybe-double))

(defn- get-architectural-components [report-spec]
  (let [{:keys [path-fn]} report-spec
        architectural-knowledge-loss (csv-conversions/csv->maps (path-fn paths/arch-knowledge-loss-csv))]
    (->> (:architectural-components report-spec)
       (map #(assoc %
               :matcher (file-patterns/path-glob-pred (:pattern %) false)
               :component-mastery (get-component-mastery (:component %) architectural-knowledge-loss))))))

(defn- key-personnel-and-knowledge-distribution-data [report-spec]
  (let [{:keys [path-fn]} report-spec
        dashboard (csv-conversions/optional-csv->map (path-fn paths/dashboard-csv))
        architectural-components (get-architectural-components report-spec)
        social-system-map (map #(merge % (get-component % architectural-components)) (get-social-system-data path-fn))]
    {:key-personnel
       {:abandoned-code      (parse-nilable-percentage (:knowledgelosspercentage dashboard))
        :key-personnel       (parse-nilable-int (:nkeypersonnel dashboard))
        :key-personnel-ratio (parse-nilable-percentage (:nkeypersonnelpercentage dashboard))}
     :knowledge-data-by-component
        (get-files-group-by-component architectural-components social-system-map)}))

(defn- management-data [report-spec]
  (let [{:keys [path-fn analysis-time]} report-spec
        dashboard (csv-conversions/optional-csv->map (path-fn paths/dashboard-csv))
        risks-for-notes (read-optional-json (path-fn paths/risks-for-notes-json) [])
        system-hotspots-health (csv-conversions/optional-csv->maps (path-fn paths/system-hotspots-health-csv))
        classified-hotspot-stats (read-optional-json (path-fn paths/classified-spot-stats-csv) [])]
    {:system-health (system-health dashboard risks-for-notes system-hotspots-health classified-hotspot-stats analysis-time)
     :code-health-overall (-> (code-health-overall path-fn)
                              health/dashboard-parameters->code-health)
     :analysis-dashboard-url (:analysis-dashboard-url report-spec)}))

(defn- management-datas [report-specs-by-project]
  (into {} (map
             (fn [[project-name report-spec]] [project-name (management-data report-spec)])
             report-specs-by-project)))

(defn for-management [report-specs-by-project options out]
  (pdf-reports/management-report (management-datas report-specs-by-project) options out))

(defn for-technical-health-overview [report-spec options out]
  (pc/pdf (pdf-reports/technical-health-overview (technical-health-and-trend-data report-spec) options) out))

(defn for-trend-report [report-spec options out]
  (pc/pdf (pdf-reports/trend-report (technical-health-and-trend-data report-spec) options) out))

(defn for-key-personnel-and-knowledge-distribution [report-spec options out]
  (pc/pdf (pdf-reports/key-personnel-and-knowledge-distribution (key-personnel-and-knowledge-distribution-data report-spec) options) out))
