(ns codescene.features.reports.common.styling
  (:require
   [clojure.java.io :as io]
   [clojure.string :as string]
   [clojure.string :as str]
   [codescene.features.reports.common.style-def :as style-def]
   [tick.core :as t])
  (:import (java.net URL)))


(defn ^URL load-resource [dir name]
  (let [path (str dir "/" name)]
    (io/resource path)))

(defn- instant->month-string [the-instant]
  (let [month-texts {1 "Jan" 2 "Feb" 3 "Mar" 4 "Apr" 5 "May" 6 "Jun"
                     7 "Jul" 8 "Aug" 9 "Sep" 10 "Oct" 11 "Nov" 12 "Dec"}
        month-number (.getValue (t/month the-instant))]
    (get month-texts month-number)))

(defn- instant->long-month-string [the-instant]
  (let [month-texts {1 "January" 2 "February" 3 "March" 4 "April" 5 "May" 6 "June"
                     7 "July" 8 "August" 9 "September" 10 "October" 11 "November" 12 "December"}
        month-number (.getValue (t/month the-instant))]
    (get month-texts month-number)))

(defn- instant->day-of-week-string [the-instant]
  (let [day-texts {1 "Mon" 2 "Tue" 3 "Wed" 4 "Thu" 5 "Fri" 6 "Sat" 7 "Sun"}
        day-number (.getValue (t/day-of-week the-instant))]
    (get day-texts day-number)))

(defn instant->printable-string [the-instant]
  (when (t/instant? the-instant)
    (let [month-string (instant->month-string the-instant)
          day-string (str (t/day-of-month the-instant))
          year-string (str (t/year the-instant))]
      (format "%s %s, %s" month-string day-string year-string))))


(defn instant->printable-string-with-time [the-instant]
  (when (t/instant? the-instant)
    (let [day-of-month-string (str (t/day-of-month the-instant))
          day-of-week-string (instant->day-of-week-string the-instant)
          month-string (instant->month-string the-instant)
          year-string (str (t/year the-instant))
          hour-string (format "%02d" (t/hour the-instant))
          minute-string (format "%02d" (t/minute the-instant))]
      (format "%s %s %s, %s at %s:%s"
              day-of-week-string month-string day-of-month-string
              year-string
              hour-string minute-string))))

(defn- instant->printable-string-long-month-without-year [the-instant]
  (when (t/instant? the-instant)
    (let [month-string (instant->long-month-string the-instant)
          day-string (str (t/day-of-month the-instant))]
      (format "%s %s" month-string day-string))))

(defn- instant->printable-string-long-month-with-year [the-instant]
  (when (t/instant? the-instant)
    (let [without-year-string (instant->printable-string-long-month-without-year the-instant)
          year-string (str (t/year the-instant))]
      (format "%s, %s" without-year-string year-string))))

(defn interval->printable-string [beginning end]
  (when (and (t/instant? beginning) (t/instant? end))
    (let [same-year (= (t/year beginning) (t/year end))
          beginning-string (if same-year
                             (instant->printable-string-long-month-without-year beginning)
                             (instant->printable-string-long-month-with-year beginning))
          end-string (instant->printable-string-long-month-with-year end)]
      (format "%s - %s" beginning-string end-string))))

(defn- instant->printable-string-short-month-without-year [the-instant]
  (when (t/instant? the-instant)
    (let [month-string (instant->month-string the-instant)
          day-string (str (t/day-of-month the-instant))]
      (format "%s %s" month-string day-string))))

(defn- instant->printable-string-short-month-with-year [the-instant]
  (when (t/instant? the-instant)
    (let [without-year-string (instant->printable-string-short-month-without-year the-instant)
          year-string (subs (str (t/year the-instant)) 2)]
      (format "%s, '%s" without-year-string year-string))))
(defn interval->short-printable-string [beginning end]
  (when (and (t/instant? beginning) (t/instant? end))
    (let [same-year (= (t/year beginning) (t/year end))
          beginning-string (if same-year
                             (instant->printable-string-short-month-without-year beginning)
                             (instant->printable-string-short-month-with-year beginning))
          end-string (instant->printable-string-short-month-with-year end)]
      (format "%s - %s" beginning-string end-string))))

(defn number->printable-string [the-number]
  (let [int-part (if the-number (int the-number) 0)]
    (format "%,d" int-part)))

(defn number->printable-string-100k [the-number]
  (if (and the-number (>= the-number 100000))
    (let [number-part (/ the-number 1000)
          number-string (number->printable-string number-part)]
      (format "%sk" number-string))
    (number->printable-string the-number)))

(defn number->cutoff-string [the-number]
  (if (number? the-number)
    (if (< the-number 10)
      (format "%.1f" (double the-number))
      (format "%.0f" (double the-number)))
    "-"))


(comment
 (format "%.0f" 10)
 (double "2")
  (number->cutoff-string 9.00000)
  (number->printable-string nil)
  (number->printable-string-100k nil))

(defn string->cutoff-string [the-string max-length]
  (if (> (count the-string) max-length)
    (let [string-part (subs the-string 0 (- max-length 3))]
      (str string-part "..."))
    the-string))


(defn score->icon [img-dir score]
  (let [img-file (case score
                   0 "Risky.png"
                   1 "Attention.png"
                   2 "Healthy.png"
                   "NA.png")]
    (load-resource img-dir img-file))
  )

(defn sub-score->icon [img-dir score]
  (let [img-file (case score
                   0 "Unhealthy-sub.png"
                   1 "Problematic-sub.png"
                   2 "Healthy-sub.png"
                   "NA.png")]
    (load-resource img-dir img-file)))

(defn score->cell-style [score]
  (let [score-styles {2 :c-border-healthy
                      1 :c-border-attention
                      0 :c-border-risky}]
    (get score-styles score :c-border-na)))


(defn score->color [score]
  (let [score-colors {2 style-def/healthy-color
                      1 style-def/attention-color
                      0 style-def/risky-color}]
    (get score-colors score style-def/na-color)))

(defn unavailable? [value]
  (cond (number? value) (zero? value)
        (string? value) (or (string/blank? value) (nil? (parse-double value)) (zero? (parse-double value)))
        :else true))