(ns codescene.features.reports.code-health.code-health-overview
  (:require
    [codescene.features.reports.pdf-helpers :as ph]
    [codescene.features.reports.graphics.functions :as graphics]
    [codescene.features.reports.common.pdf-elements :as pdf-elements]
    [codescene.features.reports.common.styling :as styling]))


(defn- n-files-with-fallback
  [n-files percentage total-files]
  (cond 
    (number? n-files)
    n-files
    ;; This fallback is for older analysis
    (every? number? [percentage total-files])
    (let [fraction (/ percentage 100)]
      (* fraction total-files))))

(defn- info-text [percentage application-code-loc]
  (if (every? number? [percentage application-code-loc])
    (let [percentage-int (Math/round (double percentage))
          lines-of-code-int (Math/round (double (* percentage-int application-code-loc 0.01)))
          lines-of-code-string (styling/number->printable-string lines-of-code-int)]
      (format "%d%% (%s %s)" percentage-int lines-of-code-string "LOC"))
    "-"))

(defn- graphics-table [{:keys [red-code yellow-code green-code] :as _code-health}]
  (ph/table
    [:cho-graphics-table] [80]
    (ph/row
      (ph/cell
        [:cho-graphics-cell]
        {:background-layer-fn
         (ph/draw-using-g2d-fn
           (graphics/code-health-overview-doughnut-fn
             {:red-value red-code :yellow-value yellow-code :green-value green-code}))}))))

(defn- info-table [{:keys [code-health] :as dashboard}]
  (let [application-code-loc (:application-code-loc dashboard)
        {:keys [red-code  yellow-code green-code ]} code-health
        red-info-text (info-text red-code application-code-loc)
        yellow-info-text (info-text yellow-code application-code-loc)
        green-info-text (info-text green-code application-code-loc)]
    (ph/table
      [] nil
      (ph/row
        (pdf-elements/info-with-score-color :c-border-healthy "Healthy: Green code" green-info-text)
        (pdf-elements/info-with-score-color :c-border-attention "Problematic: Yellow code" yellow-info-text)
        (pdf-elements/info-with-score-color :c-border-risky "Unhealthy: Red code" red-info-text)))))

(defn pdf-element [{:keys [file-summary dashboard]} project-name {:keys [img-dir]}]
  (let [formatted-project-name (styling/string->cutoff-string project-name 31)
        tip-icon (styling/load-resource img-dir "Tip.png")
        bottom-text "Reduce cost and deliver new features faster by improving code health. Unhealthy code has 15x more defects, 2x longer development time and 9x lower certainty in task completion."]
    (ph/table
      [:c-card-content-top-table] nil
      (ph/row
        (ph/cell
          []
          (ph/table
            [] [3 14]
            (ph/row (ph/cell [] (graphics-table (:code-health dashboard)))
                    (ph/cell [:cho-righthand-cell]
                             (ph/paragraph [:c-miniheader] "CODE HEALTH OVERVIEW")
                             (ph/paragraph [:cho-project-name] formatted-project-name)
                             [:spacer 1]
                             (info-table dashboard))))))
      (ph/row (ph/cell [] [:spacer 1] (ph/line [:c-line]) [:spacer 1]))
      (ph/row (ph/cell [] (ph/table [] [1 20] (ph/row
                                                (ph/cell [:cho-tip-cell] (ph/image [:c-health-image] tip-icon))
                                                (ph/cell [] (ph/paragraph [:c-subheader] bottom-text)))))))))
