(ns codescene.features.pm-data.jira.jira-provider
  "A PM Data Provider for fetching data from JIRA.
   Uses cache for fetching data. 
   This means that a single instance of the Provider will only ever fetch data once,
   and a new instance (with a new cache) is needed for fetching updated data."
  (:require [codescene.pm-data.pm-data-provider :refer [PmDataProvider]]
            [codescene.features.pm-data.jira.jira-fetcher :as f]
            [codescene.features.pm-data.jira.jira-cache :as jc]
            [codescene.features.pm-data.jira.jira-ticket-id-extractor :as x]
            [codescene.features.pm-data.pm-data-configuration :refer [PmDataConfiguration]]
            [codescene.features.pm-data.pm-data-provider :refer [-create-pm-data-provider -uses-supported-work-types]]
            [clojure.string :as str]
            [taoensso.timbre :as log]))

;;; Ideally, we would trim this when saving the api-url. However, this
;;; has already been set on many projects.
(defn- trim-trailing-slash [url]
  (if (str/ends-with? url "/")
    (subs url 0 (dec (count url)))
    url))

(defn- with-href [{:keys [api-url] :as _provider-def} issues]
  (let [->href #(format "%s/browse/%s" (trim-trailing-slash api-url) %)]
    (map #(assoc % :href (->href (:id %)))
         issues)))

(defn- fetch-project [{:keys [api-client] :as provider-def} project since]
  (let [issues (->> (jc/fetch-issues since api-client project provider-def)
                    (with-href provider-def))]
    (log/infof "Import %d issues from Jira project %s" (count issues) project)
    issues))

(defn- get-pm-data
  [provider-def {:keys [since] :as _pm-data-context}]
  (let [{:keys [external-project-ids]} provider-def
        issues (->> external-project-ids
                    (mapcat #(fetch-project provider-def % since))
                    (into []))]
    {:issues issues
     :provider-def provider-def}))

(defn- get-configuration-data [{:keys [api-client api-url use-labels-as-work-types] :as _provider-def} pm-data-context]
  (log/info "Fetching configuration data from Jira")
  {:work-types  (if use-labels-as-work-types
                  (f/fetch-labels api-client api-url)
                  (f/fetch-issue-types api-client))
   :fields      (f/fetch-number-fields api-client)
   :transitions (f/fetch-transitions api-client)
   :projects    (f/fetch-projects api-client api-url)})

(defn- validate-settings
  [{:keys [api-client password] :as _provider-def}]
  (when (seq password)
    (f/fetch-myself api-client))
  nil)

(deftype JiraProvider [provider-def pm-data-context]
  PmDataProvider
  (-make-ticket-id-extractor [_this _repo] (x/make-ticket-id-extractor provider-def))
  (-get-pm-data [_this] (get-pm-data provider-def pm-data-context))
  PmDataConfiguration
  (-get-configuration-data [_this] (get-configuration-data provider-def pm-data-context))
  (-validate-settings [_this] ( validate-settings provider-def)))

(defmethod -create-pm-data-provider "jira"
  [provider-def pm-data-context]
  (->JiraProvider provider-def pm-data-context))

(defmethod -uses-supported-work-types "jira"
  [{:keys [use-labels-as-work-types] :as _provider-def}]
  (boolean use-labels-as-work-types))

(comment
  (def repo-url "https://github.com/dotnet/aspnetcore.git")
  (def since (evolutionary-metrics.trends.dates/string->date "2020-02-25"))
  (def context {:repo-urls [repo-url]
                :since since})
  (def api-client (codescene.features.client.api/->ExtraProperties
                   {:api-url "https://empear.atlassian.net"}
                   [(System/getenv "JIRA_USER") (System/getenv "JIRA_PASSWORD")]))
  (def provider-def {:external-project-ids ["ENTERPRISE"]
                     :cost-field ""
                     :cost-unit "issues"
                     :rename-work-types [["Refactoring" "R"] ["Bug" "B"]]
                     :defect-and-failure-labels ["B"]
                     :work-in-progress-transition-names ["In Progress"]
                     :work-done-transition-names ["Done"]
                     :api-client api-client
                     :api-url "https://empear.atlassian.net"})
  (get-configuration-data provider-def context)
  (validate-settings provider-def)
  (get-pm-data provider-def since)
  )
