(ns codescene.features.pm-data.bitbucket.pr-data-provider
  (:require [codescene.features.pm-data.pr-data-provider :refer [-create PrDataProvider]]
            [codescene.features.pm-data.bitbucket.cache :as cache]
            [codescene.features.pm-data.bitbucket.fetcher :as fetcher]
            [codescene.url.url-utils :as url]
            [evolutionary-metrics.trends.dates :as dates]))

(defn- fetch-pull-requests
  [api-client repo-url since]
  (if-let [repo-info (url/repo-url->repo-info repo-url :bitbucket true)]
    (cache/fetch-pull-requests since api-client repo-info)
    []))

(defn- validate-settings
  "Validate settings by trying to fetch the latest PR data"
  [{:keys [api-client] :as _provider-def} {:keys [repo-urls] :as _pr-data-context}]
  (when-let [repo-info(-> repo-urls first (url/repo-url->repo-info :bitbucket false))]
    (fetcher/fetch-pull-requests (dates/start-of-day) api-client repo-info)))

(defn- get-pull-requests
  [{:keys [api-client] :as _provider-def} {:keys [repo-urls] :as _pr-data-context} since]
  (->> repo-urls
       (mapcat #(fetch-pull-requests api-client % since))
       (into [])))

(deftype BitBucketProvider [provider-def pr-data-context]
  PrDataProvider
  (-validate-settings [_this] (validate-settings provider-def pr-data-context))
  (-get-pull-requests [_this since] (get-pull-requests provider-def pr-data-context since)))

(defmethod -create "bitbucket"
  [provider-def pr-data-context]
  (->BitBucketProvider provider-def pr-data-context))

(comment
  (def api-client {:api-client [(System/getenv "BITBUCKET_USER")
                                (System/getenv "BITBUCKET_APP_PASSWORD")]})
  (def context {:repo-urls ["git@bitbucket.org:empear/analysis-target.git"]})
  (validate-settings api-client context)
  (get-pull-requests api-client context nil)
  )
