(ns codescene.features.factors.routes
  "Makes the four factors kpi and trend data accessible for power users who
  want to create their own dashboards.
  Focuses on ease of:
  - plotting a nice kpi over time graph
  - discovering sub kpi endpoints
  - programmatic consumption with simple code

  Documentation will be primarily conveyed in the swagger metadata, not in the
  data itself."
  (:require
    [clojure.spec.alpha :as s]
    [codescene.features.factors.api :as api]
    [codescene.factors.code-health.presentation :as code-health]
    [codescene.factors.delivery.presentation :as delivery]
    [codescene.factors.knowledge.presentation :as knowledge]
    [codescene.factors.team-code.presentation :as tca]
    [codescene.features.api.core :as api-core]
    [codescene.features.api.privileges :as api-privileges]
    [codescene.features.api.spec.analysis :as api-analysis-spec]
    [codescene.features.factors.spec.trend :as trend-spec]
    [codescene.features.factors.spec.code-health-sub-kpi-trend :as code-health-sub-kpi-trend-spec]
    [codescene.features.factors.spec.development-time-trend :as development-time-trend-spec]
    [codescene.features.factors.spec.percent_trend :as percent-trend-spec]
    [compojure.api.sweet :refer [context GET undocumented]]
    [spec-tools.core :as st]))

(s/def ::delivery-scope #{"day" "week" "month" "year"})

(declare start)
(declare end)

(defn- knowledge-routes
  [system project-id start end]
  (context "/knowledge" []
    (GET "/" []
      :responses {200 {:schema (st/spec {:spec ::trend-spec/kpi-time-series
                                         :swagger/example [{:date "2021-04-23" :kpi 0.68} {:date "2021-04-24" :kpi 0.73}]})
                       :description "A trend consisting of samples with analysis date and kpi."}}
      :description "A trend describing the knowledge distribution of your project"
      (let [raw-trend (api/latest-raw-trend knowledge/kpi-key system project-id)]
        (api/with-date-filter start end (partial api/filter-trend raw-trend))))
    (context "" []
      :responses {200 {:schema (st/spec {:spec ::percent-trend-spec/kpi-time-series
                                         :swagger/example [{:date "2021-04-23" :kpi 68} {:date "2021-04-24" :kpi 31}]})
                       :description "A trend consisting of samples with analysis date and percentage kpi."}}
      (GET "/code-familiarity" []
        :description "A low number indicates that a significant amount of code is written by former contributors who have since left the organization."
        (let [raw-trend (api/latest-raw-trend knowledge/kpi-key system project-id)
              sub-kpi-trend (api/select-sub-kpi-presentation-value-from-trend knowledge/code-familiarity-key raw-trend)]
          (api/with-date-filter start end (partial api/filter-trend sub-kpi-trend))))
      (GET "/knowledge-islands" []
        :description "A high number indicates that you have key personnel risks where the organization can lose significant knowledge if one or two developers leave."
        (let [raw-trend (api/latest-raw-trend knowledge/kpi-key system project-id)
              sub-kpi-trend (api/select-sub-kpi-presentation-value-from-trend knowledge/knowledge-islands-key raw-trend)]
          (api/with-date-filter start end (partial api/filter-trend sub-kpi-trend)))))))

(defn- delivery-routes
  [system project-id start end]
  (context "/delivery" []
    (GET "/" []
      :description "A trend describing the delivery efficiency of your project."
      :responses {200 {:schema (st/spec {:spec ::trend-spec/kpi-time-series
                                         :swagger/example [{:date "2021-04-23" :kpi 0.68} {:date "2021-04-24" :kpi 0.73}]})
                       :description "A trend consisting of samples with analysis date and kpi."}}
      (let [raw-trend (api/latest-raw-trend delivery/kpi-key system project-id)
            flattened-trend (api/flatten-delivery-trend raw-trend)]
        (api/with-date-filter start end (partial api/filter-trend flattened-trend))))
    (GET "/development-time" []
      :description "A short development time is a pre-requisite for speed to market."
      :responses {200 {:schema (st/spec {:spec ::development-time-trend-spec/kpi-time-series
                                         :swagger/example [{:date "2021-04-23" :kpi 1456.9811320754718} {:date "2021-04-24" :kpi 60}]})
                       :description "A trend consisting of samples with analysis date and average development time in minutes."}}
      (let [raw-trend (api/latest-raw-trend delivery/kpi-key system project-id)
            flattened-trend (api/flatten-delivery-trend raw-trend)
            sub-kpi-trend (api/select-sub-kpi-presentation-value-from-trend delivery/development-time-key flattened-trend)]
        (api/with-date-filter start end (partial api/filter-trend sub-kpi-trend))))
    (GET "/unplanned-work" []
      :description "Unplanned work is bug fixes and unexpected rework that the organisation didn't anticipate."
      :responses {200 {:schema (st/spec {:spec            ::percent-trend-spec/kpi-time-series
                                         :swagger/example [{:date "2021-04-23" :kpi 3} {:date "2021-04-24" :kpi 60}]})
                       :description "A trend consisting of samples with analysis date and percentage of unplanned work."}}
      (let [raw-trend (api/latest-raw-trend delivery/kpi-key system project-id)
            flattened-trend (api/flatten-delivery-trend raw-trend)
            sub-kpi-trend (api/select-sub-kpi-presentation-value-from-trend delivery/unplanned-work-key flattened-trend)]
        (api/with-date-filter start end (partial api/filter-trend sub-kpi-trend))))))

(defn- code-health-average-result [system project-id start end]
  (let [raw-trend (api/latest-raw-trend code-health/kpi-key system project-id)
        sub-kpi-trend (api/select-sub-kpi-from-trend code-health/average-code-health-key raw-trend)]
    (api/with-date-filter start end (partial api/filter-trend sub-kpi-trend))))

(defn- code-health-routes
  [system project-id start end]
  (context "/code-health" []
    (undocumented
      (GET "/" []
        ;;we return same data as the average endpoint,
        ;; see: https://app.clickup.com/t/9015696197/CS-1175
        (code-health-average-result system project-id start end)))
    (context "" []
      :responses {200 {:schema (st/spec {:spec ::code-health-sub-kpi-trend-spec/kpi-time-series
                                         :swagger/example [{:date "2021-04-23" :kpi 9.96} {:date "2021-04-24" :kpi 9.62}]})
                       :description "A trend consisting of samples with analysis date and kpi."}}
      (GET "/average" []
        :description "Average code health is the weighted average health of all the files in the codebase. The code health is weighted based on the lines of code in each file; the larger the file, the larger the impact. Maximum score is 10 (healthy, green code) and lowest score is 1.0 (red code with severe technical debt)."
        (code-health-average-result system project-id start end))
      (GET "/hotspots" []
        :description "Hotspot code health is a weighted average of the code health in the system wide hotspots. The code health is weighted based on the lines of code in each file; the larger the file, the larger the impact. Maximum score is 10 (healthy, green code) and lowest score is 1.0 (red code with severe technical debt)."
        (let [raw-trend (api/latest-raw-trend code-health/kpi-key system project-id)
              sub-kpi-trend (api/select-sub-kpi-from-trend code-health/hotspots-code-health-key raw-trend)]
          (api/with-date-filter start end (partial api/filter-trend sub-kpi-trend))))
      (GET "/worst-performer" []
        :description "The worst performer score represents the code health of one or more files with the lowest code health score in the codebase. Maximum score is 10 (healthy, green code) and lowest score is 1.0 (red code with severe technical debt)."
        (let [raw-trend (api/latest-raw-trend code-health/kpi-key system project-id)
              sub-kpi-trend (api/select-sub-kpi-from-trend code-health/worst-performer-key raw-trend)]
          (api/with-date-filter start end (partial api/filter-trend sub-kpi-trend)))))))

(defn- tca-routes
  [system project-id start end]
  (context "/team-code-alignment" []
    (GET "/" []
      :responses {200 {:schema (st/spec {:spec ::trend-spec/kpi-time-series
                                         :swagger/example [{:date "2021-04-23" :kpi 0.68} {:date "2021-04-24" :kpi 0.73}]})
                       :description "A trend consisting of samples with analysis date and kpi."}}
      :description "A trend describing the team code alignment in your project"
      (let [raw-trend (api/latest-raw-trend tca/kpi-key system project-id)]
        (api/with-date-filter start end (partial api/filter-trend raw-trend))))
    (context "" []
      :responses {200 {:schema (st/spec {:spec ::percent-trend-spec/kpi-time-series
                                         :swagger/example [{:date "2021-04-23" :kpi 68} {:date "2021-04-24" :kpi 31}]})
                       :description "A trend consisting of samples with analysis date and percentage kpi."}}
      (GET "/team-cohesion" []
        :description "Strong team cohesion means that the team carries a meaning from an architectural perspective."
        (let [raw-trend (api/latest-raw-trend tca/kpi-key system project-id)
              sub-kpi-trend (api/select-sub-kpi-presentation-value-from-trend tca/team-cohesion-key raw-trend)]
          (api/with-date-filter start end (partial api/filter-trend sub-kpi-trend))))
      (GET "/team-coupling" []
        :description "Loosely coupled teams minimizes coordination and the risk of conflicting changes to the code."
        (let [raw-trend (api/latest-raw-trend tca/kpi-key system project-id)
              sub-kpi-trend (api/select-sub-kpi-presentation-value-from-trend tca/team-coupling-key raw-trend)]
          (api/with-date-filter start end (partial api/filter-trend sub-kpi-trend)))))))

(defn sub-routes
  [system project-id]
  (context "/kpi-trend" []
    :tags ["KPI trends"]
    :middleware [#(api-core/wrap-authorize-project system % project-id #{api-privileges/restapi-read})]
    :query-params [{start :- (st/spec {:spec ::api-analysis-spec/date_type
                                       :description "Format: YYYY-MM-DD. An empty date returns data since the beginning of the trend."
                                       :swagger/example "2021-04-23"}) ""}
                   {end :- (st/spec {:spec ::api-analysis-spec/date_type
                                     :description "Format: YYYY-MM-DD. An empty end date returns the trend up to the latest value."
                                     :swagger/example "2021-04-23"}) ""}]
    (code-health-routes system project-id start end)
    (delivery-routes system project-id start end)
    (knowledge-routes system project-id start end)
    (tca-routes system project-id start end)))
