(ns codescene.features.developer.core
  (:require [clojure.java.io :as io]
            [semantic-csv.core :as sc]
            [codescene.analysis.paths :as paths]
            [codescene.features.util.maps :refer [map-of]]))

(defn- authors-from-csv [analysis-path file]
  (let [path (paths/make-analysis-path-to analysis-path file)]
    (if (.exists (io/file path))
      (->> path sc/slurp-csv (mapv :author))
      [])))

(defn- make-is-configured-p
  [kw developers]
  (->> developers
       (filter kw)
       (map :name)
       set))

(defn- analyzed-devs 
  "Returns active-devs and potential-exdevs for an analysis"
  [path]
  (let [all-authors (authors-from-csv path paths/all-authors-csv)
        potential-ex-devs (authors-from-csv path paths/detected-ex-devs-warning-csv)
        active-devs (remove (set potential-ex-devs) all-authors)]
    (map-of active-devs potential-ex-devs)))

(defn potential-ex-devs 
  "Consider a dev as a potential exdev when
   - is identified as a potential exdev in one of the analyzed projects
   - is not an active dev in another project
   - is not already configured as an exdev"
  [paths developers]
  (let [analyzed-devs (map analyzed-devs paths)
        active-devs (into #{} (mapcat :active-devs) analyzed-devs)
        potential-ex-devs (into #{} (mapcat :potential-ex-devs) analyzed-devs)
        already-exdev? (make-is-configured-p :exdev developers)]
    (->> potential-ex-devs
         (remove active-devs)
         (remove already-exdev?))))

(defn teamless-authors [paths developers]
  (let [already-has-team? (make-is-configured-p :team developers)]
    (->> paths
         (mapcat #(authors-from-csv % paths/teamless-authors-csv))
         distinct
         (remove already-has-team?))))
