(ns codescene.features.delta.coverage.cov-gates
  (:require [clojure.java.jdbc :as jdbc]
            [codescene.features.components.db :as db]
            [codescene.features.project.core :as pc]
            [codescene.features.config.presets :as presets]
            [codescene.features.delta.coverage.cov-config :as cov]
            [codescene.features.delta.coverage.pr-check :as pr-check]
            [codescene.features.delta.integration.integration-db :as integration-db]
            [codescene.features.util.maps :refer [map-of]]
            [taoensso.timbre :as log]))

(defn settings-id
  "Returns ID of QG settings stored at scope"
  [db-spec scope]
  (or (get-in (presets/available-settings db-spec scope ["cov-config"]) ["cov-config" 0 :id])
      (presets/create-settings {:db-spec db-spec
                                :scope scope
                                :encryptor identity
                                :params {:category "cov-config"
                                         :label "Code Coverage Quality Gates"
                                         :cov-qg-preset "custom"
                                         :new-and-changed-code? true}
                                :html-form-input? false})))

(defn qg-presets
  "Returns global or other scoped qg presets"
  [db-spec scope]
  (let [id (settings-id db-spec scope)]
    (presets/get-settings db-spec id scope)))

(def qg-preset-desc
  (integration-db/preset-desc
    "cov-config"
    #(assoc % :qg-preset "global")
    {:qg-preset "custom"
     :new-and-changed-code? true}))

(defn get-project-settings [system project-id]
  (jdbc/with-db-transaction [tx (db/db-spec system)]
    (let [project (pc/component system project-id)
          {:keys [exclusion-filter exclude-content whitelist-content
                  code-coverage-exclusions-content] :as _proj} (pc/-filters-configuration project tx)
          {:keys [data] :as config-set} (some-> (integration-db/get-config tx project-id (constantly cov/settings-properties))
                                                (integration-db/resolve-presets tx
                                                                                {:cov-qg-preset qg-preset-desc}
                                                                                (pc/-scope project tx)))
          exclude-content-from-coverage (not-empty code-coverage-exclusions-content)
          gates (pr-check/conf->gates data)]
      (when (pr-check/check-enabled? config-set)
        (map-of project-id exclusion-filter exclude-content whitelist-content
                exclude-content-from-coverage gates)))))
