(ns codescene.features.delta.code-health.ch-config
  (:require [clojure.spec.alpha :as s]
            [codescene.features.delta.integration.integration-config :as ic]
            [codescene.features.util.maps :as maps]
            [codescene.features.config.properties :refer [enum bool sval property ival] :as p]
            [spec-tools.data-spec :as ds]))

(s/def ::github-file-comments #{"review" "annotations" "none"})
(s/def ::qg-preset #{"global" "minimal" "medium" "all" "custom" "custom-file"})

(def provider-properties
  {:github [(property :file-comments :github-file-comments (enum ::github-file-comments) "review")]
   :azure [(property :annotations? :azure-annotations (bool) true)
           (property :active-status? :azure-active-status (bool) true)
           (property :high-risk-active-comment? :azure-high-risk-active-comment (bool) false)]
   :bitbucket [(property :build? :bitbucket-build (bool) true)
               (property :report? :bitbucket-report (bool) true)
               (property :annotations? :bitbucket-annotations (bool) true)]
   :gitlab [(property :discussions? :gitlab-discussions (bool) true)
            (property :show-pending? :gitlab-show-pending (bool) false)
            (property :high-risk-active-comment? :gitlab-high-risk-active-comment (bool) false)]
   :bitbucket-server [(property :build? :bitbucket-server-build (bool) true)
                      (property :report? :bitbucket-server-report (bool) true)
                      (property :annotations? :bitbucket-server-annotations (bool) true)]
   :gerrit [(property :comments? :gerrit-comments (bool) false)]
   :github-hook []
   :plugin-gerrit []
   :none []})

(def result-options [:file-comments :annotations? :active-status? :high-risk-active-comment?
                     :build? :report? :discussions? :show-pending? :always-comment? :comments?])

(def qg-properties [(property :qg-preset :qg-preset (enum ::qg-preset) "global")
                    (property :qg-hotspot-decline? :qg-hotspot-decline (bool) false)
                    (property :qg-critical-health-rules? :qg-critical-health-rules (bool) false)
                    (property :qg-advisory-health-rules? :qg-advisory-health-rules (bool) false)
                    (property :qg-new-code-health? :qg-new-code-health (bool) false)
                    (property :qg-refactoring-goals? :qg-refactoring-goals (bool) false)
                    (property :qg-supervise-goals? :qg-supervise-goals (bool) false)
                    (property :codeowners-for-critical-code? :qg-codeowners (bool) false)])

(def qg-properties-keys (mapv :id qg-properties))
(def qg-properties-keys-wo-preset (next qg-properties-keys))

(def general-delta-properties [(property :coupling-threshold :coupling-threshold (ival) 80)
                               (property :always-comment? :always-comment (bool) false)])

(def every-provider-properties (concat general-delta-properties qg-properties))


(defn api-spec
  "API Spec for all configs"
  [provider-property-maps]
  (let [key-spec-coll (for [props (vals provider-property-maps)
                            {:keys [type-info ui-param]} props]
                        [(ds/opt (maps/->db-name ui-param)) (:api-spec type-info)])]
    ;; the name doesn't matter, but it does show up in Swagger, so we pick a nice public one
    (ds/spec {:spec (into {} key-spec-coll) :name :pr-integration/config})))

(def ch-on-prem-api-properties
  "Code Health REST API properties"
  (merge-with into provider-properties ic/onprem-properties))

(defn remove-qg-props
  "Removes qg properties from property list (except preset itself). Useful
  when updating on global preset and you don't want to overwrite existing properties."
  [properties]
  (let [p (set qg-properties-keys-wo-preset)]
    (remove (comp p :id) properties)))

(def ch-cloud-project-config-properties
  (let [all-providers (concat every-provider-properties ic/general-delta-properties)]
    (update-vals provider-properties #(into % all-providers))))