(ns codescene.features.config.properties
  (:require [clojure.spec.alpha :as s]
            [clojure.string :as str]
            [codescene.presentation.display :refer [->maybe-int]]
            [codescene.features.api.spec.analysis :as api-specs]
            [codescene.features.util.maps :refer [map-of]]
            [codescene.specs :as specs]
            [medley.core :as m]
            [taoensso.timbre :as log]))

(s/def ::ui-param keyword?)
(s/def ::param-read ifn?)
(s/def ::deser ifn?)
(s/def ::encrypt? boolean?)
(s/def ::missing-false? boolean?)
(s/def ::fix-fn (s/nilable ifn?))
(s/def ::default any?)
(s/def ::type-info (s/keys :opt-un [::missing-false? ::encrypt? ::param-read ::deser ::ser]))
(s/def ::property (s/keys :req-un [::ui-param ::default ::type-info]
                          :opt-un [::fix-fn]))
(s/def ::properties (s/map-of any? ::property))

(def standard-param-fix (fn [x] (if (string? x) (str/trim x) x)))

;; param-read fn is (fn [property-map data] v)
(defn default-param-read [{:keys [ui-param fix-fn] :or {fix-fn standard-param-fix} :as x} data]
  (if-some [e (find data ui-param)]
    (fix-fn (val e))
    ::missing))

(defn file-read [{:keys [ui-param]} params]
  (or (m/find-first seq [(some-> params (get-in [(keyword (str (name ui-param) "-file")) :tempfile]) slurp)
                         (get params ui-param)])
      ::missing))

(defn ival [] {:deser #(when % (->maybe-int %)) :api-spec integer?})

(defn file-string
  [encrypt?] {:encrypt? encrypt? :api-spec ::specs/non-empty-string :param-read file-read})

(defn enum
  [s] {:api-spec s })

(defn empty->missing [v]
  (if (empty? (str/trim v)) ::missing v))

(defn sval
  "Non-empty string value."
  ([] (sval false))
  ([encrypt?] {:encrypt? encrypt?
               :api-spec ::specs/non-empty-string
               :param-read (fn [om params]
                             (default-param-read
                               (update om :fix-fn #(cond-> empty->missing % (comp %)))
                               params))}))

(defn sval-
  "String value that can be empty."
  ([] (sval- false))
  ([encrypt?] {:encrypt? encrypt? :api-spec string?}))

(defn bool [] {:deser #(case %
                         nil nil
                         ("true" true "1") true
                         ("false" false "0") false
                         true)
               :missing-false? true
               :api-spec ::api-specs/boolean-string})

(defn property
  ([id ui-param type-info]
   (map-of id ui-param type-info))
  ([id ui-param type-info default]
   (map-of id ui-param default type-info)))

(s/fdef property
        :args (s/cat :id any? :ui-param ::ui-param :type-info ::type-info :default (s/? ::default)))

(defn from-req-params
  "Reads HTTP form params into an option -> value map.

  When data comes from UI, then html-form-input? should be set to true, since in UI
  boolean values are implemented with checkboxes which only send parameter when checked."
  [properties params try-encrypt html-form-input?]
  (reduce (fn [acc {:keys [id type-info] :as om}]
            (let [{:keys [param-read deser encrypt? missing-false?] :or {param-read default-param-read deser identity}} type-info
                  v (param-read om params)]
              (cond
                (not= v ::missing) (assoc acc id (cond-> (deser v) encrypt? try-encrypt))
                (and html-form-input? missing-false?) (assoc acc id false)
                ;(contains? om :default) (assoc acc id default)
                :else acc)))
          {}
          properties))

(defn to-params
  [properties data]
  (let [by-id (m/index-by :id properties)]
    (m/map-keys #(get-in by-id [% :ui-param] %) data)))

(defn to-db
  "Serializes data to DB format (string keys, values), but it doesn't add defaults for missing properties, those
  are added on DB read."
  [properties data]
  (let [by-id (m/index-by :id properties)]
    (reduce-kv (fn [acc k v]
                 (if-let [{:keys [type-info]} (by-id k)]
                   (assoc acc (str (symbol k)) ((:ser type-info str) v))
                   acc))
               {}
               data)))

(defn from-db
  "Reads DB config into a map."
  [properties data]
  (reduce (fn [acc {:keys [default type-info id] :as p}]
            (if-some [v (find data id)]
              (assoc acc id ((or (:deser type-info) identity) (val v)))
              (if (contains? p :default) (assoc acc id default) acc)))
          {}
          properties))

(defn config-set
  "Property sets equip properties with additional information about the properties used and
  other metadata. The key thing is state, as data exists in 3 states, :ui-params, :live (which is domain representation)
  and :db, which is serialized for DB."
  [metadata data properties state]
  (map-of metadata data properties state))

(defn to-ui-params
  "Convert a set to UI params representation"
  [{:keys [properties data state] :as config-set}]
  (case state
    :live (assoc config-set
            :data (to-params properties data)
            :state :ui-params)
    :ui-params config-set
    :db (recur (assoc config-set :data (from-db properties data) :state :live))
    nil nil))

(defn live-to-db
  "Convert a set to DB representation"
  [{:keys [properties data state] :as config-set}]
  (case state
    :live (assoc config-set
            :data (to-db properties data)
            :state :db)
    :db config-set
    nil nil))

(defn from-req-params'
  "Reads HTTP form params into an option -> value map.

  When data comes from UI, then html-form-input? should be set to true, since in UI
  boolean values are implemented with checkboxes which only send parameter when checked."
  [{:keys [properties data] :as config-set} to-state try-encrypt html-form-input?]
  (when config-set
    (case to-state
      :live (assoc config-set
              :data (from-req-params properties data try-encrypt html-form-input?)
              :state :live)
      :ui-params config-set
      :db (live-to-db (from-req-params' config-set :live try-encrypt html-form-input?)))))