(ns codescene.features.components.http
  "HTTP Client Configuration component. It allows easy configuration of
  Common timeout configuration, etc."
  (:require [clj-http.conn-mgr :as conn]
            [clojure.string :as str]
            [codescene.features.components.core :refer [find-component]]
            [codescene.features.components.mock-http :as mock]
            [medley.core :as m]
            [taoensso.timbre :as log]
            [integrant.core :as ig])
  (:import (org.apache.http.conn.routing HttpRoute)
           (org.apache.http HttpHost)
           (org.apache.http.pool PoolStats)
           (org.apache.http.impl.conn PoolingHttpClientConnectionManager)))

(defn log-max-connections-and-reset
  "Call this periodically (hourly) to record highest observed number of used connections."
  [config]
  (when-let [counter (:max-leased config)]
    (log/infof "Highest HTTP Client connection lease: %s/%s"
               @counter
               (when-let [^PoolingHttpClientConnectionManager conn-mgr (:connection-manager config)]
                 (-> conn-mgr (.getMaxTotal))))
    (reset! counter 0)))

(defn track-max-connections-tick
  "Call this regularly to update the max number of connections in use observed."
  [config]
  (when-let [conn-mgr (:connection-manager config)]
    (let [^PoolStats total-stats (.getTotalStats conn-mgr)]
      (swap! (or (:max-leased config) (atom 0)) #(max % (.getLeased total-stats)))
      (when (= (.getMax total-stats) (.getLeased total-stats))
        (log/warnf "All %s(%s) HTTP Client pool connections leased! Routes: %s"
                   (.getMax total-stats)
                   (.getPending total-stats)
                   (vec (for [route (.getRoutes conn-mgr)
                              :let [route-stats (.getStats conn-mgr route)]]
                          (format "%s %s(%s)/%s"
                                  route
                                  (.getLeased route-stats)
                                  (.getPending route-stats)
                                  (.getMax route-stats)))))))))

(defn- create-conn-manager [constructor props max-connections-per-host]
  (when (seq props)
    (let [cm (constructor props)
          route (fn [url]
                  (HttpRoute. (HttpHost/create url) nil (str/starts-with? url "https://")))]
      (doseq [[url n] max-connections-per-host]
        (.setMaxPerRoute cm (route url) n))
      cm)))

;; produces a IDeref which enables a component that has part of http client configuration happen on the fly
(defmethod ig/init-key ::config [_ {:keys [conn-manager async-conn-manager base-opts max-connections-per-host]}]
  (delay (m/assoc-some base-opts
           :max-leased (atom 0)
           :connection-manager (create-conn-manager conn/make-reusable-conn-manager
                                                    conn-manager
                                                    max-connections-per-host)
           :async-conn-manager (create-conn-manager conn/make-reusable-async-conn-manager
                                                    async-conn-manager
                                                    max-connections-per-host))))

(derive ::config :codescene/component)

(defmethod ig/halt-key! ::config [_ config-ideref]
  (doseq [[k v] @config-ideref]
    (case k
      :connection-manager (conn/shutdown-manager v)
      :async-conn-manager (conn/shutdown-manager v)
      :http-client (.close v)
      :async-http-client (.close v)
      nil)))

(defn client-config [system] (some-> (find-component system ::config) deref))

(defmethod ig/init-key ::mocked [_ mock-rules]
  (delay (mock/mocked-client mock-rules)))

(derive ::mocked ::config)
