(ns codescene.features.code-coverage.parsers.bullseye-parser
  (:require [codescene.features.code-coverage.parsers.xml-utils :refer [filter-on-tag]]
            [codescene.features.code-coverage.parser :refer [Parser]]
            [codescene.features.code-coverage.parsers.utils :as utils]
            [clojure.java.io :as io]
            [clojure.data.xml :as xml]
            [clojure.string :as str]
            [medley.core :as m]))

(def ^:private supported-metrics [:line-coverage
                                  :method-coverage
                                  :condition-decision-coverage
                                  :decision-coverage])

(defn attrs->coverage-data [attrs covered-attr total-attr]
  (let [covered-str (get attrs covered-attr)
        total-str (get attrs total-attr)]
    (when (and covered-str total-str)
      (utils/->entry (Integer/parseInt covered-str) (Integer/parseInt total-str)))))

(defn- condition-decision-coverage [attrs]
  (attrs->coverage-data attrs :cd_cov :cd_total))

(defn- decision-coverage [attrs]
  (attrs->coverage-data attrs :d_cov :d_total))

(defn- method-coverage [attrs]
  (attrs->coverage-data attrs :fn_cov :fn_total))

(defn- lines-matching 
  "Returns a set with all lines matching the given criteria (covered or uncovered).
   
   Interpreting the data:
   BullsEye has the concept of probes:
   
   full	      Fully covered (executed)	✅ Covered
   true	      Only the true branch taken	⚠️ Partially covered
   false	    Only the false branch taken	⚠️ Partially covered
   none	      Not executed at all	❌ Uncovered
   (missing)	Not executed (implicitly)	❌ Uncovered"
  [covered-uncovered-fn?
   probes]
   (->> probes 
        (filter #(covered-uncovered-fn? (:event %)))
        (map :line)
        set))

(defn- line-coverage
  "we count as total every line probe and if the event is not 'none' then the line is counted as covered
   see: https://github.com/SonarOpenCommunity/sonar-cxx/wiki/sonar.cxx.bullseye.reportPaths
   and https://github.com/SonarOpenCommunity/sonar-cxx/blob/95ea2aec5d0df9374435ae89bd12ffde8734e7a2/cxx-sensors/src/main/java/org/sonar/cxx/sensors/coverage/bullseye/BullseyeParser.java"
  [parse-options src-el]
  (let [probes (->> (:content src-el)
                    (filter-on-tag :fn)
                    (mapcat :content)
                    (filter-on-tag :probe)
                    (map :attrs))
        count-as-string (comp str count)
        covered   (lines-matching (partial not= "none") probes)
        uncovered (lines-matching (partial = "none")    probes)
        ln-covered (count-as-string covered)
        ln-total (->> (map :line probes) distinct count-as-string)
        the-coverage (attrs->coverage-data {:ln_covered ln-covered :ln_total  ln-total} :ln_covered :ln_total)]
    (merge the-coverage
           (when (utils/needs-detailed-line-coverage? parse-options)
             {:details {:covered covered
                        :uncovered uncovered}}))))

(defn ->coverage-entry 
  [parse-options
   folders 
   {:keys [attrs] :as src-el}]
  (let [{:keys [name]} attrs]
  (m/assoc-some {:path (str/join "/" (conj folders name))
                   :name name}
                  :condition-decision-coverage (condition-decision-coverage attrs)
                  :decision-coverage (decision-coverage attrs)
                  :method-coverage (method-coverage attrs)
                  :line-coverage (line-coverage parse-options src-el))))

(defn- parse-folder-el [parse-options parent-folders {:keys [attrs content] :as _folder-el}]
  (let [{:keys [name]} attrs
        folders (conj parent-folders name)
        src-els (filter-on-tag :src content)
        sub-folder-els (filter-on-tag :folder content)]
    (concat (map (partial ->coverage-entry parse-options folders) src-els)
            (mapcat (partial parse-folder-el parse-options folders) sub-folder-els))))

(defn- read-coverage* [reader parse-options]
  (let [xml (-> reader (xml/parse :namespace-aware false :support-dtd false))]
    (->> xml :content
         (filter-on-tag :folder)
         (mapcat #(parse-folder-el parse-options [] %)))))

(defn- read-coverage [f parse-options]
  (with-open [r (io/reader f)]
    (doall (read-coverage* r parse-options))))

(defn ->Parser []
  (reify Parser
    (-read-coverage [this reader parse-options] (read-coverage* reader parse-options))
    (-supported-metrics [this] supported-metrics)
    (-id [this] "bullseye")
    (-name [this] "BullsEye")))

(comment
  (def f "./test/codescene/features/code_coverage/testdata/bullseye.xml")
  (def parse-options {:cli-command "check"})
  (read-coverage f parse-options)
  )
