(ns codescene.features.code-coverage.custom-metrics-provider
  (:require [codescene.custom-metrics.custom-metrics-provider :refer [CustomMetricsProvider]]
            [evolutionary-metrics.mining.file-patterns :as file-patterns]
            [hotspots-x-ray.recommendations.code-properties.core :as code-properties]
            [medley.core :as mc]
            [clojure.string :as str]
            [clojure.set :as set]))

(defn- make-scope->repo-id-lookup
  "Creates a lookup for repo-path and repo-url by hotspot 'scope'"
  [repo-path->repo-id]
  (->> (keys repo-path->repo-id)
       file-patterns/make-scope->repo-lookup
       ;; now we have a lookup for paths, but we want id:s...
       (mc/map-vals repo-path->repo-id)))

(defn- coverage-map [{:keys [file-changed? covered total]} estimated-code-size]
  (let [cc-data? (some? total)
        current? (false? file-changed?)]
    {:covered (or covered 0)
     ;; Estimation is impossible for files without code health score.
     :total   (or total estimated-code-size 0)
     :cc-data? cc-data?
     ;; Consider valid files without CC data (whether we can estimate or not), otherwise check whether it's current
     :valid?  (or (not cc-data?) current?)}))

(defn- ->score [{:keys [scope->repo-id coverage-lookup exclude?]} code-properties-fn module metric]
  (let [[scope file-path] (file-patterns/de-scope module)
        repo-id (scope->repo-id scope)]
    (when-not (exclude? repo-id file-path)
      (let [coverage-for-file (get-in coverage-lookup [repo-id file-path metric])
            estimated-code-size (-> module code-properties-fn code-properties/app-code-size)]
        {metric (coverage-map coverage-for-file
                              estimated-code-size)}))))

(defn- score-hotspot [coverage-context code-properties-fn metrics {:keys [module]}]
  (let [scores (->> metrics
                    (keep (partial ->score coverage-context code-properties-fn module))
                    (into {}))]
    (when (not-empty scores)
      {:module module
       :scores scores})))

(defn- score-hotspots [coverage-context code-properties-fn metrics hotspots]
  (keep (partial score-hotspot coverage-context code-properties-fn metrics) hotspots))

(def metric-id->name {:line-coverage "Line Coverage"
                      :branch-coverage "Branch Coverage"
                      :function-coverage "Function Coverage"
                      :condition-coverage "Condition Coverage"
                      :condition-decision-coverage "Condition/Decision Coverage"
                      :decision-coverage "Decision Coverage"
                      :method-coverage "Method Coverage"
                      :sequence-point-coverage "Sequence Point Coverage"
                      :statement-coverage "Statement Coverage"})

(defn- coverage-metric
  [metric]
  {:id metric
   :name (or (-> metric keyword metric-id->name) metric)
   :type "coverage"
   :unit "%"
   :min-value 0
   :max-value 100
   :is-positive true})

(defn- get-coverage
  "We switched from a single coverage number to a map with several entries.
   the old format will be ignored"
  [file-data metric]
  (let [coverage-data (get file-data metric)]
    (when (map? coverage-data)
      coverage-data)))

(defn metric-reduce-fn
  "Reduce fn over the metric m"
  [{:keys [repo files-changed]} {:keys [path] :as file-data} acc m]
  (let [file-changed? (set files-changed)]
    (if-let [coverage (get-coverage file-data m)]
      (assoc-in acc [repo path m] (merge {:file-changed? (-> path file-changed? some?)} coverage))
      acc)))

(defn coverage-reduce-fn
  "reduce fn over the actual coverage data in each def"
  [{:keys [metric repo] :as coverage-data-def} acc {:keys [path] :as file-data}]
  (let [metrics (conj #{:line-coverage} (keyword metric))]
    (reduce (partial metric-reduce-fn coverage-data-def file-data) acc metrics)))

(defn data-def-reduce-fn
  "reduce fn over stored coverage data definitions"
  [acc {:keys [delayed-data] :as coverage-data-def}]
  (reduce (partial coverage-reduce-fn coverage-data-def) acc (deref delayed-data)))

(defn- make-coverage-lookup
  "Creates a coverage lookup [repo path metric]->value,
   based on coverage-data-defs sorted by creation date"
  [coverage-data-defs]
  ;; Sort by date to make newer ones override earlier ones
  (reduce data-def-reduce-fn {} (sort-by :created-at coverage-data-defs)))

(defn- metric-ids
  "return the distinct metric-ids from coverage-data-defs
  we add line-coverage if we have at least one metric"
  [coverage-data-defs]
  (-> (map :metric coverage-data-defs)
      seq
      (some-> (conj "line-coverage"))
      distinct))

(defn- exclusion-paths-from [code-coverage-exclusions-content]
  (if (not-empty code-coverage-exclusions-content)
    (->> (str/split code-coverage-exclusions-content #";")
         (map str/trim))
    []))


(defn- root-star-pattern?
  [exc-path]
  (let [root (first (file-patterns/de-scope exc-path))]
    (cond
      (= "*" root) true
      (= "**" root) true
      (and
        ;; Ignore patterns with double-stars in the root, mixed with
        ;; characters: this just gets too complicated.  Should be
        ;; documented.
        (not (re-matches #".*\*\*.*" root))
        ;; If it's made it this far, then we have at least one,
        ;; non-consecutive star + something else. We don't care what
        ;; else is here.
        (str/includes? root "*")) true
      :else
      false)))

(defn- matches-scope?
  [scope pattern]
  (let [scope-of-pattern  (first (file-patterns/de-scope pattern))
        scope-pattern (file-patterns/globs-matcher [scope-of-pattern] false)]
    (if (contains? #{"**" "*"} scope-of-pattern)
      true
      (scope-pattern scope))))

;(matches-scope? "repo*/blah" "repo-to-match")

(defn- mixed-star-patterns-for-scope
  [scope mixed-star-paths]
   (->> mixed-star-paths
        (filter (partial matches-scope? scope))
        (map file-patterns/de-scope)))

(defn- exclusion-lookup
  "Returns a map of repo-ids to path matchers. This ends up being a bit
  complicated because most matching is per-repo, ie. inside a single
  repo, but we also want to allow matching against multiple
  repos. Three different approaches are allowed:

  - **/my/pattern.clj 
  - */my/pattern.clj
  - codescene-*/my/pattern.clj
  
  This function applies these to any repo that matches.

  Patterns like this won't work however and will be silently ignored:

  - codescene-**/my/nested/pattern.clj
"
  [exclusion-paths scope->repo-id]
  (let [repo-root-star-paths (filter root-star-pattern?  exclusion-paths)
        repo-specific-paths (remove root-star-pattern? exclusion-paths)
        repo-specific-matchers (->> repo-specific-paths
                                    (map file-patterns/de-scope)
                                    (group-by first)
                                    (map (fn [[k vals]]
                                           (let [possible-mixed-star-patterns (mixed-star-patterns-for-scope k repo-root-star-paths)]
                                             [(scope->repo-id k)
                                              (file-patterns/globs-matcher (map second (concat vals possible-mixed-star-patterns)) false)])))
                                    ;; We don't want any nil keys (might only happen with test data)
                                    (filter (comp some? first))
                                    (into {}))
        repos-without-matchers (set/difference (set (vals scope->repo-id))
                                               (set (keys repo-specific-matchers)))
        reverse-scope-lookup (set/map-invert scope->repo-id)]
    (if (not-empty repos-without-matchers)
      (->> repos-without-matchers
           (map (fn [repo]
                  (when (not-empty repo-root-star-paths)
                    [repo (file-patterns/globs-matcher (map second (mixed-star-patterns-for-scope (get reverse-scope-lookup repo) repo-root-star-paths)) false)])))
           (into {})
           (merge repo-specific-matchers))
      repo-specific-matchers)))

(defn exclusion-matcher
  "Returns a function that returns `true` when a file should be excluded."
  [code-coverage-exclusions-content scope->repo-id]
  (let [lookup (-> code-coverage-exclusions-content
                   exclusion-paths-from
                   (exclusion-lookup scope->repo-id))]
    (fn [repo path]
      (when-let [glob-matcher (get lookup repo)]
        (glob-matcher path)))))

(defn ->CustomMetricsProvider [{:keys [coverage-data-defs validation-results]} project-params]
  (let [{:keys [repo-path->repo-id code-coverage-exclusions-content]} project-params
        metrics (metric-ids coverage-data-defs)
        scope->repo-id (make-scope->repo-id-lookup repo-path->repo-id)
        exclude? (exclusion-matcher code-coverage-exclusions-content scope->repo-id)
        coverage-lookup (make-coverage-lookup coverage-data-defs)
        coverage-context {:coverage-lookup coverage-lookup :scope->repo-id scope->repo-id :exclude? exclude?}]
    (reify CustomMetricsProvider
      (-id [_this] "code-coverage")
      (-metrics [_this] [] (map coverage-metric metrics))
      (-score-hotspots [_this code-properties-fn hotspots]
        (score-hotspots coverage-context code-properties-fn (map keyword metrics) hotspots))
      (-validation-results [_this] validation-results))))



(comment
  (require '[clj-time.core :as tc])
  (require '[clojure.edn :as edn])


(defn coverage-data
  [path]
  (edn/read-string (slurp path)))

 (first repo-id->head-revision)

 (def cov (delay  (coverage-data "/Users/joseph/Empear/src/codescene/features/test/codescene/features/code_coverage/testdata/code-coverage.edn")))
 (first @cov)

  (def lu (make-coverage-lookup [{:delayed-data cov :created-at 1 :metric "statement-coverage" :repo "repo" :files-changed #{}}]))

  (-> [{:repo "git@github.com:knorrest/react.git",
        :delayed-data (delay (edn/read-string (slurp "/Users/kalle/codescene/dev/code-coverage-data/code-coverage-10.edn")))
        :metric "statement-coverage"
        :commit-sha "f9dddcbbb1c0b73f974e78b9488927b778630683"
        :created-at (.toDate (tc/date-time 2024 5 2))}
       {:repo "git@github.com:knorrest/react.git",
        :commit-sha "f9dddcbbb1c0b73f974e78b9488927b778630680"
        :created-at (.toDate (tc/date-time 2024 5 1))}
       {:repo "git@github.com:knorrest/react.git",
        :commit-sha "f9dddcbbb1c0b73f974e78b9488927b778630683"
        :created-at (.toDate (tc/date-time 2024 5 3))}]
      make-coverage-lookup)

  (def repo-id->head-revision {"github.com/knorrest/react" "f9dddcbbb1c0b73f974e78b9488927b778630682"})
  (def coverage-data-defs [{:delayed-data (delay (edn/read-string (slurp "/Users/kalle/codescene/dev/code-coverage-data/code-coverage-10.edn")))
                            :repo "github.com/knorrest/react",
                            :commit-sha "f9dddcbbb1c0b73f974e78b9488927b778630682"
                            :created-at (java.util.Date.)
                            :metric "statement-coverage"}])
  (def repo-path->repo-id {"/Users/kalle/codescene/dev/repos/95335ce6e7ee635b246ea47e4932390b9ad8e937/react" "github.com/knorrest/react"})

  (def repo-id->head-revision {"github.com/knorrest/commercial-management--ndc--offer-availability--microservice" "ea022367693c0ff89b747dfff72d27f7bdc376b0"})
  (def coverage-data-defs [{:delayed-data (delay (edn/read-string (slurp "/Users/kalle/Downloads/code-coverage-256.edn")))
                            :repo "github.com/knorrest/commercial-management--ndc--offer-availability--microservice",
                            :commit-sha "ea022367693c0ff89b747dfff72d27f7bdc376b0"
                            :created-at (java.util.Date.)
                            :metric "line-coverage"}])
  (def repo-path->repo-id {"/var/codescene/repos/2700205/d1e1a973-ba25-4d2d-9368-dca9cc9fdc7e/repos/commercial-management--ndc--offer-availability--microservice" "github.com/knorrest/commercial-management--ndc--offer-availability--microservice"})

  (-> (make-coverage-lookup coverage-data-defs)
      (get "github.com/knorrest/commercial-management--ndc--offer-availability--microservice")
      (get "src/main/java/com/iberia/offer/core/model/utils/FlightNumberUtils.java"))
  (metric-ids coverage-data-defs)
  (make-scope->repo-id-lookup repo-path->repo-id)
  (score-hotspots (make-coverage-lookup coverage-data-defs)
                  (make-scope->repo-id-lookup repo-path->repo-id)
                  [:line-coverage]
                  [{:module "commercial-management--ndc--offer-availability--microservice/src/main/java/com/iberia/offer/core/model/request/OfferAvailabilityRequest.java"}]))
