(ns codescene.features.code-coverage.core
  (:require [clojure.set :refer [map-invert] :as set]
            [clojure.string :as str]
            [codescene.features.code-coverage.data-validator :as data-validator]
            [codescene.features.code-coverage.custom-metrics-provider :refer [->CustomMetricsProvider]]
            [codescene.url.url-utils :as url]
            [evolutionary-metrics.mining.vcs :as vcs]
            [medley.core :as m]
            [clojure.edn :as edn]
            [taoensso.timbre :as log]))

(defmulti current-git-head-for-repo
  "With this multimethod, we can inject our own function for testing,
  with having to supply a git repository."
  (fn [git-client _repo-path]
    (if (fn? git-client) :function :command)))

(defmethod current-git-head-for-repo :command [git-client repo-path]
  (vcs/git-head-revision-in git-client repo-path ))

(defmethod current-git-head-for-repo :function [injected-fn repo-path]
  (injected-fn {:repo-path repo-path
                :vcs-equivalent :git-head-revision-in}))

(defn applicable-coverage-data
  ([{:keys [git-client repo-id->repo-path  read-coverage-data-fn]} coverage-data-defs]
   (->> coverage-data-defs
        (group-by :repo)
        (map (fn [[repo defs]]
               (when-let [repo-path (repo-id->repo-path repo)]
                 (let [analyzed-commit (current-git-head-for-repo git-client repo-path)]
                   (data-validator/applicable-coverage-data-for
                    {:git-client  git-client
                     :repo-path repo-path
                     :repo repo         
                     :read-coverage-data-fn read-coverage-data-fn
                     :analyzed-commit analyzed-commit}
                    defs)))))
        (reduce (fn [result current]
                  {:coverage-data-defs (concat (:coverage-data-defs result) (:coverage-data-defs current))
                   :validation-results (concat (:validation-results result) [(:validation-result current)])})
                {:coverage-data-defs [] :validation-results []}))))


(defn- log-malformed-paths
  [data-defs-with-path-errors]
  (doseq [{:keys [data-path path-tree repo]} data-defs-with-path-errors]
    (when-let [error (:error path-tree)]
      (log/warnf  "Malformed path %s fetching coverage data for %s: %s. Skipping" data-path repo error)))
  data-defs-with-path-errors)


(defn deleteable-coverage-data
  [coverage-data-defs read-coverage-data-fn]
  (let [pre-flight-data-defs (data-validator/data-defs-for-priority-filtering read-coverage-data-fn coverage-data-defs)
        error-path-tree? (fn [{:keys [path-tree]}] (:error path-tree))
        undeletable-ids-because-unfetchable (->> pre-flight-data-defs
                                                 (filter error-path-tree?)
                                                 log-malformed-paths
                                                 (map :id)
                                                 set)
        ids-to-keep (->> pre-flight-data-defs
                         (sort-by :created-at)
                         (remove error-path-tree?)
                         data-validator/replace-superseded-data-defs
                         (map :id)
                         set
                         (set/union undeletable-ids-because-unfetchable))
        deletable? (fn [{:keys [id]}] (not (contains? ids-to-keep id)))
        deleting (filter deletable? coverage-data-defs)]
    (log/infof "Planning to delete %d coverage definitions for repo %s and [id / date] %s"
               (count ids-to-keep)
               (if (empty? coverage-data-defs) "<no repo>" (:repo (first coverage-data-defs)))
               (str/join ", "  (map (comp str (juxt :id :created-at)) deleting)))
    deleting))

 (defn incomplete-coverage-data
   [{:keys [repo subpath format] :as _metadata} coverage-data-defs]
   (filter #(and (nil? (:data-path %))
                 (= repo (:repo %))
                 (= subpath (:subpath %))
                 (= format (:format %)))
           coverage-data-defs))

(defn custom-metrics-provider
  [git-client coverage-data-defs {:keys [repo-path->remote-url code-coverage-exclusions-content  read-coverage-data-fn]}]
  (let [repo-path->repo-id (m/map-vals url/repo-url->repo-id repo-path->remote-url)
        repo-id->repo-path (map-invert repo-path->repo-id)
        coverage-data-params {:git-client git-client
                              :repo-id->repo-path repo-id->repo-path
                              :read-coverage-data-fn read-coverage-data-fn}
        applicable-coverage-data (applicable-coverage-data coverage-data-params coverage-data-defs)]
    (->CustomMetricsProvider applicable-coverage-data {:repo-path->repo-id repo-path->repo-id
                                                       :code-coverage-exclusions-content code-coverage-exclusions-content})))


