(ns codescene.features.api.core
  (:require [codescene.features.components.auth :as auth]
            [codescene.features.components.project :as project]
            [codescene.features.util.api :as api-utils]))

(defn api-component [system]
  (:codescene.features.api/api-component system))

(defn api-developer-settings [system]
  (-> system api-component :developer-settings))

(defn api-projects [system]
  (-> system api-component :project))

(defn api-analyses [system]
  (-> system api-component :analysis))

(defn api-delta-analyses [system]
  (-> system api-component :delta-analysis))

(defn api-auth [system]
  (-> system api-component :auth))

(defn api-env [system]
  (-> system api-component :env))

(defn api-licensing [system]
  (-> system api-component :licensing))

(defn api-code-coverage-storage [system]
  (-> system api-component :code-coverage-storage))

(defn api-analytics [system]
  (-> system api-component :analytics))

(defn api-ff [system]
  (-> system api-component :ff))

(defn api-root
  "return api-root set in the env or default-api-root"
  [system]
  (-> (api-env system)
      (get :api-root api-utils/default-api-root)))

(defn wrap-authorize-any
  "verify if authenticated user has the required privileges"
  [system handler required-privileges]
  (fn
    ([request]
     (let [{:keys [privileges]} (auth/user (api-auth system) request)
           processed-required-privileges (auth/process-required-privileges (api-auth system) required-privileges)]
       (if (some processed-required-privileges privileges)
         (handler request)
         (api-utils/forbidden (format "Forbidden access, missing privileges: %s" required-privileges)))))
    ([request respond raise]
     (let [{:keys [privileges]} (auth/user (api-auth system) request)
           processed-required-privileges (auth/process-required-privileges (api-auth system) required-privileges)]
       (if (some processed-required-privileges privileges)
         (handler request respond raise)
         (respond (api-utils/forbidden (format "Forbidden access, missing privileges: %s" required-privileges))))))))

(defn wrap-authorize-project
  "verify if authenticated user has the required privileges and access to the project-id"
  [system handler project-id required-privileges]
  (fn [request]
    (let [handler-with-authorize (wrap-authorize-any system handler required-privileges)
          project (project/project-by-id (api-projects system) project-id)]
      (cond
        (not project)
        (api-utils/not-found (format "There's no project matching the id: %s" project-id))

        (not (auth/has-project-access? (api-auth system) request project-id))
        (api-utils/forbidden (format "Forbidden access to project with id: %s" project-id))

        :else
        (handler-with-authorize request)))))

(defn wrap-can-use-feature
  "verify that the authenticated user can access the specified feature"
  [system handler feature]
  (fn
    ([request]
     (if (auth/can-use-feature? (api-auth system) request feature)
         (handler request)
         (api-utils/forbidden (format "Your license doesn't include access to %s feature" feature))))
    ([request respond raise]
     (if (auth/can-use-feature? (api-auth system) request feature)
         (handler request respond raise)
         (api-utils/forbidden (format "Your license doesn't include access to %s feature" feature))))))
