(ns tools.docker-migration
  "Tool to aid in migrating an existing CodeScene installation
   running as a standalone JAR to a Docker based installation.
   The challenge is to take care of all of the data the CodeScene
   keeps outside of the database. For now the strategy is to discard
   that data in a controlled fashion, because it can be regenerated 
   by running an analysis again."
  (:require [cacsremoteservice.features.analysis.analysis-feature :as analysis]
            [cacsremoteservice.database.db :as db]
            [cacsremoteservice.features.delta-analysis.delta :as delta]
            [cacsremoteservice.features.project.project-feature :as project]
            [clojure.java.jdbc :as jdbc]
            [taoensso.timbre :as log]
            [cacsremoteservice.database.migrations :as db-migrations]))

(defn local-git-repo?
  "Checks if the project's remote is a local path. These projects are not migratable."
  [project]
  (empty? (:local-path-for-remotes project)))

(defn migratable?
  [project]
  (not (local-git-repo? project)))

(defn reset-result-paths!
  [tx {:keys [id name] :as _project}]
  (log/infof "Updating result paths for %s" name)
  (project/reset-result-paths! tx id))

(defn clear-old-results!
  [tx {:keys [name id] :as _project}]
  (log/infof "Clearing old results for %s" name id)
  (analysis/delete-analyses-for-project! tx id)
  (delta/delete-analyses-for! tx id))

(defn remove-project!
  [tx {:keys [name id] :as _project}]
  (log/infof "Removing project %s" name)
  (project/delete-configuration-of! tx id false))
  
(defn simple-migration!
  "The simple migration does not copy all old analysis results and repos
  on disk to the docker volume, but it does preserve all settings and
  other analysis results that are stored in the DB.

  Preprocess:
  1) Check if there are projects with local git paths 
  2) Remove those projects

  Migration steps:
  1) Update base paths to cloned repos and analysis results
  2) Clear all old results

  The user will then have to rerun an analysis to access the projects."
  [db-spec]
  (jdbc/with-db-transaction [tx db-spec]
    (let [all-projects (project/all-projects tx)]
      (doseq [project (remove migratable? all-projects)]
        (remove-project! tx project))
      (doseq [project (filter migratable? all-projects)]
        (reset-result-paths! tx project)
        (clear-old-results! tx project)))))

(defn print-project-names
  [projects]
  (doseq [name (map :name projects)]
    (println name)))

(defn check
  "Check if we can access the DB and list what will be done in the migration.

  Projects that are not migratable will be displayed."
  [db-spec]
  (let [projects (project/all-projects db-spec)
        project-with-local-repos (filter local-git-repo? projects)
        migratable-projects (filter migratable? projects)]
    (when (seq project-with-local-repos)
      (println "Will not migrate projects with local git repos:")
      (print-project-names project-with-local-repos))
    (if (seq migratable-projects)
      (do
        (println "Will migrate the following projects:")
        (print-project-names migratable-projects))
      (println "No migratable projects found!"))))

(defn- db-init []
  (let [db-spec (db/init)]
    (db-migrations/migrate db-spec)
    db-spec))

(comment
  (def db-spec (db-init))
  (simple-migration! db-spec)
  (check db-spec))

(defn print-help
  []
  (println "docker-migration.sh [run | check]"))

(defn migrate
  [cmd]
  (cond
    (= "run" cmd) (simple-migration! (db-init))
    (= "check" cmd) (check (db-init))
    :else (print-help))
   ;; The following is to avoid losing buffered output
  (flush)
  (shutdown-agents))
