grammar ScalaNestedComplexity;
options {superClass=hotspots_x_ray.languages.InterruptibleParser;}

// This grammar detects islands of nested complexity.

method
:
	expression* EOF
;

expression : complexity
           | block_expression
           | anything;

complexity: if_clause
          | else_if_clause
          | else_clause
          | for_loop
          | for_generator
          | foreach_multiline
          | foreach_singleline
          | while_loop
          | for_comprehension
          | match_clause;

block_expression: OPENING_BRACE expression*? CLOSING_BRACE;

anything: ~(OPENING_BRACE | CLOSING_BRACE);

if_clause: IF some_condition (multi_line_conditional_expression | plain_line);
else_if_clause: ELSE IF some_condition (multi_line_conditional_expression | plain_line);
else_clause: ELSE (multi_line_conditional_expression | plain_line);

multi_line_conditional_expression: OPENING_BRACE expression*? CLOSING_BRACE;
plain_line: expression;

for_loop: FOR LEFT_PAREN for_loop_part TO for_loop_part RIGHT_PAREN (multi_line_conditional_expression | plain_line);
for_loop_part: ~(TO)+?; // each part is optional

for_generator: FOR some_condition multi_line_conditional_expression;

foreach_multiline: FOREACH multi_line_conditional_expression;

foreach_singleline: FOREACH LEFT_PAREN;

for_comprehension: FOR ~(TO | YIELD)+? YIELD;

match_clause: match_condition MATCH multi_line_conditional_expression;
match_condition: ID (LEFT_PAREN ~(RIGHT_PAREN)*? RIGHT_PAREN)?;

while_loop: WHILE some_condition (multi_line_conditional_expression | plain_line);

some_condition: LEFT_PAREN conditions+? RIGHT_PAREN;
conditions : conditional_operator
           | ~(LEFT_PAREN | RIGHT_PAREN | OPENING_BRACE | IF | ELSE | FOR | WHILE | TO | MATCH)
           | LEFT_PAREN conditions*? RIGHT_PAREN;

conditional_operator: OPERATORS;

fragment ESCAPED : '\\\\' | '\\"';
LITERAL : '"' ( ESCAPED | ~('\n'|'\r') )*? '"';

IF: 'if';
ELSE: 'else';
FOR: 'for';
FOREACH: 'foreach';
TO: 'to';
YIELD: 'yield';
WHILE: 'while';
MATCH: 'match';

OPERATORS: '&&' | '||';

ID : [a-zA-Z_][a-zA-Z0-9_]*;

LEFT_PAREN: '(';
RIGHT_PAREN: ')';
OPENING_BRACE: '{';
CLOSING_BRACE: '}';

Whitespace : [ \t]+ -> skip;

BlockComment: '/*' .*? '*/' -> skip;
LineComment: '//' ~[\r\n]* -> skip;

NEWLINE : '\r'? '\n' -> skip;

ANY_CHAR : .; // Put this lexer rule last to give it the lowest precedence
