grammar JavaNestedComplexity;

// This grammar detects islands of nested complexity.

method
:
	expression* EOF
;

expression : complexity
           | block_expression
           | anything;

complexity: if_clause
          | else_clause
          | switch_clause
          | enhanced_for_loop
          | for_loop
          | while_loop
          | do_while_loop;

block_expression: OPENING_BRACE expression*? CLOSING_BRACE;

anything: ~(OPENING_BRACE | CLOSING_BRACE);

if_clause: IF some_condition (multi_line_conditional_expression | plain_line);
else_clause: ELSE (multi_line_conditional_expression | plain_line);

switch_clause: SWITCH some_condition multi_line_conditional_expression;

multi_line_conditional_expression: OPENING_BRACE expression*? CLOSING_BRACE;
plain_line: ~OPENING_BRACE ~(SEMICOLON)*? SEMICOLON;

enhanced_for_loop: FOR LEFT_PAREN enhanced_for_loop_part COLON for_loop_condition RIGHT_PAREN (multi_line_conditional_expression | plain_line);
enhanced_for_loop_part:  ~(SEMICOLON | COLON)+?;

for_loop: FOR LEFT_PAREN for_loop_part SEMICOLON for_loop_part SEMICOLON for_loop_condition RIGHT_PAREN (multi_line_conditional_expression | plain_line);
for_loop_part: ~(SEMICOLON | COLON)*?; // each part is optional
for_loop_condition: conditions*?;

while_loop: WHILE some_condition (multi_line_conditional_expression | plain_line);
do_while_loop: DO multi_line_conditional_expression WHILE some_condition;

some_condition: LEFT_PAREN conditions+? RIGHT_PAREN;
conditions :conditional_operator
           | ~(LEFT_PAREN | RIGHT_PAREN | SEMICOLON | OPENING_BRACE | IF | ELSE | FOR | WHILE | DO | SWITCH)
           | LEFT_PAREN conditions*? RIGHT_PAREN;

conditional_operator: CONDITIONAL_OPERATORS;

fragment ESCAPED : '\\\\' | '\\"';
LITERAL : '"' ( ESCAPED | ~('\n'|'\r') )*? '"';

// copied from JavaSharedRulesLexer
fragment MULTI_LINE_STRING_DELIMITER: '"""';
MULTI_LINE_STRING: MULTI_LINE_STRING_DELIMITER .*? MULTI_LINE_STRING_DELIMITER;

IF: 'if';
ELSE: 'else';
FOR: 'for';
WHILE: 'while';
DO: 'do';
SWITCH: 'switch';

CONDITIONAL_OPERATORS: '&&' | '||';

ID : [a-zA-Z_][a-zA-Z0-9_]*;

LEFT_PAREN: '(';
RIGHT_PAREN: ')';
OPENING_BRACE: '{';
CLOSING_BRACE: '}';
COLON: ':';
SEMICOLON: ';';

Whitespace : [ \t]+ -> skip;

BlockComment: '/*' .*? '*/' -> skip;
LineComment: '//' ~[\r\n]* -> skip;

NEWLINE : '\r'? '\n' -> skip;

ANY_CHAR : .; // Put this lexer rule last to give it the lowest precedence