
-- :name select-analytical-tags :? :*
-- Returns all tags
select * from analytical_tags;

-- :name select-analytical-tag-by-id :? :1
-- Returns a tag by ID
select * from analytical_tags
where id = :id;

-- :name select-analytical-tag-hierarchy :? :*
-- Returns all tag hierarchy relationships
select * from analytical_tag_hierarchy;

-- :name select-analytical-tag-assignments :? :*
-- Returns all tag assignments
select * from analytical_tag_assignments;

-- :name select-analytical-units-by-types :? :*
-- Returns all analytical units of specific types
select * from analytical_units
where type in (:v*:types);

-- :name select-analytical-units-by-types-and-projects :? :*
-- Returns all analytical units of specific types within specific projects
select * from analytical_units
where type in (:v*:types)
and project_id in (:v*:project-ids);

  -- :name select-analytical-units-by-type :? :*
-- Returns all analytical units of specific type
select * from analytical_units
where type = :type;

-- :name select-analytical-unit-by-id :? :1
-- Returns an analytical unit by ID
select * from analytical_units
where id = :id;

-- :name select-analytical-tags-by-analytical-unit :? :*
-- Returns all tags for a specific analytical unit
select t.* from analytical_tags t
join analytical_tag_assignments ta on t.id = ta.tag_id
where ta.analytical_unit_id = :analytical-unit-id;

-- :name select-descendant-analytical-tags :? :*
-- Returns all descendant tags of a given tag using recursive CTE
with recursive descendants(id) as (
  select child_tag_id as id
  from analytical_tag_hierarchy
  where parent_tag_id = :tag-id

  union all

  select th.child_tag_id
  from analytical_tag_hierarchy th
  join descendants d on th.parent_tag_id = d.id
)
select id from descendants

-- :name insert-analytical-tag! :insert
-- Inserts a new tag and returns the generated ID
insert into analytical_tags (name)
values (:name);

-- :name insert-analytical-tag-hierarchy! :insert
-- Inserts a tag hierarchy relationship
insert into analytical_tag_hierarchy (parent_tag_id, child_tag_id)
values (:parent-tag-id, :child-tag-id);

-- :name insert-analytical-unit! :insert
-- Inserts a new analytical unit and returns the generated ID
insert into analytical_units (name, type, project_id)
values (:name, :type, :project-id);

-- :name insert-analytical-tag-assignment! :insert
-- Assigns a tag to an analytical unit
insert into analytical_tag_assignments (tag_id, analytical_unit_id)
values (:tag-id, :analytical-unit-id);

-- :name select-analytical-tag-assignment :? :1
-- Checks if a tag assignment exists
select * from analytical_tag_assignments
where tag_id = :tag-id
and analytical_unit_id = :analytical-unit-id;

-- :name delete-analytical-tag-assignment! :! :n
-- Removes a specific tag assignment
delete from analytical_tag_assignments
where tag_id = :tag-id
and analytical_unit_id = :analytical-unit-id;

-- :name delete-all-analytical-tag-assignments! :! :n
-- Deletes all tag assignments
delete from analytical_tag_assignments;

-- :name delete-all-analytical-tag-hierarchy! :! :n
-- Deletes all tag hierarchy relationships
delete from analytical_tag_hierarchy;

-- :name delete-all-analytical-units! :! :n
-- Deletes all analytical units
delete from analytical_units;

-- :name delete-all-analytical-tags! :! :n
-- Deletes all tags
delete from analytical_tags;

-- :name delete-analytical-tag! :! :n
-- Deletes a specific tag by ID
delete from analytical_tags
where id = :id;

-- :name select-analytical-units-by-types-and-tags :? :*
-- Returns all analytical units of specific types having tags
select au.*
from analytical_tag_assignments ta
join analytical_units au on au.id = ta.analytical_unit_id
where au.type in (:v*:types)
and ta.tag_id in (:v*:tags)
group by ta.analytical_unit_id
having :sql:having ;

-- :name select-analytical-tags-by-names :? :*
-- Returns tags by their names
select * from analytical_tags
where name in (:v*:names);
