(ns codescene.features.tag-queries.tag-db
  (:require [hugsql.core :as hugsql]
            [clojure.string :as str]
            [medley.core :as m]))

(hugsql/def-db-fns "codescene/features/tag_queries/tag_queries.sql")

(defn- extract-generated-key
  "Extracts the generated key from a HugSQL insert result.
  Handles different result formats from different databases."
  [result]
  (or (:generated_key result)
      (get-in result [:1 :generated_key])
      (:id result)))

(defn analytical-tags
  [db-spec]
  (select-analytical-tags db-spec))

(defn analytical-tag-assignments
  [db-spec]
  (select-analytical-tag-assignments db-spec))

(defn- project-tags-by-project-id
  [db-spec]
  (let [all-projects (select-analytical-units-by-type db-spec {:type "project"})]
    (into {}
          (map (fn [project]
                 [(:project_id project)
                  (select-analytical-tags-by-analytical-unit db-spec {:analytical-unit-id (:id project)})])
               all-projects))))

(defn- add-tags-to-unit
  [db-spec project-tags-map unit]
  (let [direct-tags (select-analytical-tags-by-analytical-unit db-spec {:analytical-unit-id (:id unit)})
        project-tags (get project-tags-map (:project_id unit) [])
        ;; Combine and deduplicate tags by ID
        all-tags (vals (m/index-by :id (concat direct-tags project-tags)))]
    (assoc unit :tags all-tags)))

(defn analytical-units
  [db-spec types]
  (let [units (select-analytical-units-by-types db-spec {:types types})
        project-tags-map (project-tags-by-project-id db-spec)]
    (mapv (partial add-tags-to-unit db-spec project-tags-map) units)))

(defn descendant-analytical-tags
  [db-spec tag-id]
  (select-descendant-analytical-tags db-spec {:tag-id tag-id}))

(defn analytical-tags-with-descendants
  [db-spec tag-ids]
  (mapv (fn [tag-id]
          (conj (set (map :id (descendant-analytical-tags db-spec tag-id))) tag-id))
        tag-ids))

(defn add-analytical-tag
  ([db-spec name]
   (add-analytical-tag db-spec name nil))
  ([db-spec name parent-tag-id]
   (let [result (insert-analytical-tag! db-spec {:name name})
         tag-id (extract-generated-key result)]
     (when parent-tag-id
       (insert-analytical-tag-hierarchy! db-spec {:parent-tag-id parent-tag-id
                                                   :child-tag-id tag-id}))
     tag-id)))

(defn add-analytical-unit
  [db-spec name type project-id]
  (let [result (insert-analytical-unit! db-spec {:name name
                                                  :type type
                                                  :project-id project-id})]
    (extract-generated-key result)))

;; @CodeScene(disable:"Bumpy Road Ahead")
(defn assign-analytical-tag
  [db-spec tag-id analytical-unit-id]
  (let [tag (select-analytical-tag-by-id db-spec {:id tag-id})
        unit (select-analytical-unit-by-id db-spec {:id analytical-unit-id})]
    (when-not tag
      (throw (ex-info "Tag not found" {:tag-id tag-id})))
    (when-not unit
      (throw (ex-info "Analytical unit not found" {:analytical-unit-id analytical-unit-id})))
    (insert-analytical-tag-assignment! db-spec {:analytical-unit-id analytical-unit-id
                                                :tag-id tag-id})
    true))

(defn unassign-analytical-tag
  [db-spec tag-id analytical-unit-id]
  (let [assignment (select-analytical-tag-assignment db-spec {:tag-id tag-id
                                                              :analytical-unit-id analytical-unit-id})]
    (when-not assignment
      (throw (ex-info "Tag assignment not found" {:tag-id tag-id
                                                  :analytical-unit-id analytical-unit-id})))
    (delete-analytical-tag-assignment! db-spec {:tag-id tag-id
                                                :analytical-unit-id analytical-unit-id})
    true))

(defn- having-condition
  [tag-ids]
  (format "sum(case when ta.tag_id in (%s) then 1 else 0 end) > 0" (str/join "," tag-ids)))

(defn analytical-units-with-tags
  "Returns all analytical units of any of the specified types having all of the specified tags.
  Having a tag means having the exact tag OR one of it's descendants"
  [db-spec types tag-ids]
  (if (and (seq types) (seq tag-ids))
    (let [expanded-tag-groups (analytical-tags-with-descendants db-spec tag-ids)
          all-tags (vec (set (mapcat identity expanded-tag-groups)))
          having (str/join " and " (map having-condition expanded-tag-groups))]
      (println types all-tags having )
      (select-analytical-units-by-types-and-tags db-spec {:types (vec types)
                                                  :tags all-tags
                                                  :having having}))
    []))

(defn analytical-units-in-projects
  "Returns all analytical units of the specified types within the specified projects,
   with their tags (both direct and inherited from project)."
  [db-spec types project-ids]
  (if (and (seq types) (seq project-ids))
    (select-analytical-units-by-types-and-projects db-spec {:types (vec types)
                                                            :project-ids (vec project-ids)})
    []))
  

(defn analytical-tag-names-to-ids
  [db-spec names]
  (if (empty? names)
    []
    (->> (select-analytical-tags-by-names db-spec {:names names})
         (mapv :id))))

(defn delete-analytical-tag
  [db-spec tag-id]
  (delete-analytical-tag! db-spec {:id tag-id}))

(comment
  (def db-spec @knorrest.db/conn)

  (delete-all-analytical-units! db-spec)
  (delete-all-analytical-tags! db-spec)

  (analytical-tags db-spec)
  (analytical-units db-spec [ "team"])

  (->> (analytical-tag-names-to-ids db-spec [ "fe" ])
       (analytical-units-with-tags db-spec ["project" "component"])
       (map :name))
  )

