(ns codescene.features.tag-queries.core
  "This ns is the main API to the tag-queries functionality"
  (:require [codescene.features.tag-queries.tag-db :as tag-db]))

(defn- project-descentants
  "Return all non-project analytical units in projects with the specified tag-ids.
   (This is where sub-project units inherit tags set on project)"
  [db-spec types tag-ids]
  (let [projects (tag-db/analytical-units-with-tags db-spec ["project"] tag-ids)]
    (tag-db/analytical-units-in-projects db-spec (disj (set types) "project") (map :project_id projects))))

(defn- without-project-descendants
  "Remove AUs that are inside projects where the project AUs are among the ones passed in"
  [analytical-units]
  (let [project-ids (->> analytical-units (filter #(= "project" (:type %))) (map :project_id) set)]
    (remove #(and ( project-ids (:project_id %))
                  (not= "project" (:type %))) 
            analytical-units)))

(defn analytical-units-with-tags
  "Get all analytical-units of the requested types having the specified tag-ids.
  Note that: 
  - Tags are inherited from project to sub-project units. 
    (That is - if a project has a tag, the sub-units are considered as having it too)
  - When request both project and sub-project types, units inside found project are excluded
    (if not they would overlap, which is not a good fit for aggregating..."
  [db-spec types tag-ids]
  (let [types (set types)
        units (tag-db/analytical-units-with-tags db-spec types tag-ids)]
    (->> (if ((set types) "project")
           ;; If project is in the query, we don't want to include components inside returned projects
           ;; (there shouldn't be any, if tags are set properly...)
           (without-project-descendants units)
           ;; if project is not in the query, we need to find components inside projects that match 
           (concat units (project-descentants db-spec types tag-ids)))
         distinct
         (into []))))

  
(defn analytical-tags
  "Returns a list of all analytical tags"
  [db-spec]
  (tag-db/analytical-tags db-spec))

(defn add-analytical-tag
  "Adds analytical tag and returns the tag-id"
  [db-spec name]
  (tag-db/add-analytical-tag db-spec name))

(defn delete-analytical-tag
  "Delete analytical tag by id and return true if it was successful"
  [db-spec tag-id]
  (pos? (tag-db/delete-analytical-tag db-spec tag-id)))

(defn analytical-tag-assignments
  [db-spec]
  (tag-db/analytical-tag-assignments db-spec))

(defn assign-analytical-tag
  "Assigns an analytical tag and returns true if it was successful"
  [db-spec tag-id analytical-unit-id]
  (some? (tag-db/assign-analytical-tag db-spec tag-id analytical-unit-id)))

(defn unassign-analytical-tag
  "unassigns an analytical tag and returns true if it was successful"
  [db-spec tag-id analytical-unit-id]
  (some? (tag-db/unassign-analytical-tag db-spec tag-id analytical-unit-id)))

(defn analytical-units 
  ([db-spec]
   (analytical-units db-spec ["projects" "components" "teams" "languages"]))
  ([db-spec types]
   (tag-db/analytical-units db-spec types)))

(comment 
  "
drop table analytical_tag_assignments;
drop table analytical_tag_hierarchy;
drop table analysical_tags;
drop table analytical_units;
"

"
delete from analytical_tag_assignments;
delete from analytical_tag_hierarchy;
delete from analysis_tags;
delete from analytical_units;
"
"
insert into analytical_units(name, type, project_id)
SELECT name, 'project', id
FROM projects
" 

"
insert into analytical_units(name, type, project_id)
SELECT concat('component',id,'C'), 'component', id
FROM projects

"
  
  )
