(ns codescene.features.security.pat
  (:require [buddy.core.codecs :as bc]
            [buddy.hashers :as hashers]
            [buddy.core.nonce :as nonce]
            [clojure.string :as str]
            [codescene.features.scheduling.ephemeral :as ephemeral]
            [codescene.features.util.maps :refer [->db-keys ->db-name ->kebab map-of-db]]
            [codescene.specs :as specs]
            [clojure.spec.alpha :as s]
            [clj-time.core :as tc]
            [clj-time.coerce :as tcc]
            [codescene.util.json :as json]
            [hugsql.core :as hugsql]
            [medley.core :as m]
            [taoensso.timbre :as log]))

(hugsql/def-db-fns "codescene/features/security/pat.sql")

(s/def ::hashed (specs/string-n 255))
(s/def ::user-id ::specs/id)
(s/def ::id ::specs/id)
(s/def ::label (specs/string-n 0 255))
(s/def ::used-at (s/nilable ::specs/datetime))
(s/def ::expires-at ::specs/datetime)

(s/def ::db-token (s/keys :req-un [::hashed ::user-id ::id ::used-at ::expires-at ::label]))

(defprotocol PATPersistence
  (delete-expired-tokens [this])
  (insert-token [this token additional-data] "Returns ID")
  (delete-token [this user-id token-id] "Deletes token, if user-id is provided only deletes if user ID matches.")
  (user-tokens [this user-id])
  (all-tokens [this page])
  (search-tokens [this user-id page search-term] "Searches tokens. If user-id is nil, searches all tokens (admin), otherwise searches user's tokens.")
  (update-used-at [this token-id]))

;; We don't want to use hard password hash with many iterations, because tokens are validated
;; on every request and we try to validate every key if a person has many.
;;
;; Not really needed, because you cannot dictionary attack a random 24 byte + salt token and
;; it's not feasible to bruteforce it.

;; The number of hashes done by a GPU roughly doubles every 2 years. In 2015 a GPU could
;; crank out 1.6B SHA-1 hashes per second. Back of napkin calculations for 10000 GPUs today:
;; roughly 255 * 10^12 SHA-1 hashes per second give or take a factor of 10.

;; Our key is 24 random bytes, so it has 6.2 * 10^57 different values and we'll use SHA-512 to prevent collisions by
;; shorter values.
(def hasher-options {:alg :pbkdf2+sha512 :iterations 1})

(defn- to-insertable [t additional-data]
  (-> (assoc t :additional-columns (map (comp name ->db-name key) additional-data)
               :additional-values (map val additional-data))
      (m/update-existing :expires-at tcc/to-date)
      (m/update-existing :used-at tcc/to-date)
      ->db-keys))

(defn- to-readable [t]
  (-> (->kebab t)
      (m/update-existing :expires-at tcc/from-sql-time)
      (m/update-existing :used-at tcc/from-sql-time)))

(defn token-info
  "Extract info from Bearer token"
  [tok]
  (try
    (select-keys (json/parse-string (bc/b64->str tok true)) [:id :rand])
    (catch Exception e
      (log/warn "Couldn't decode token" e))))

(defn init-cleanup [presistence]
  (ephemeral/recurring-job
    (fn [] (delete-expired-tokens presistence))
    (* 12 3600 1000)
    "Cleanup expired user tokens")
  nil)

(defn new-token
  [id]
  ;; 192-bit key
  (-> {:id id :rand (nonce/random-bytes 24)}
      json/generate-string
      bc/str->bytes
      (bc/bytes->b64-str true)))

(defn pat-token? [token] (= (set (keys (token-info token))) #{:id :rand}))

(defn active? [token] (tc/before? (tc/now) (:expires-at token)))

(defn create-user-token [persistence user-id expiry-days additional-data]
  (let [pure-token (new-token user-id)
        expires (tc/plus (tc/now) (tc/days expiry-days))
        token {:user-id user-id :expires-at expires :hashed (hashers/derive pure-token hasher-options)}]
    (log/infof "Created a PAT for user=%s with additional-data %s" user-id additional-data)
    (merge token
           additional-data
           {:secret pure-token
            :id (insert-token persistence token additional-data)})))

(defn- match-db-token
  "Adds :valid and :update to DB token representation. It is valid if not expired and
  matching the submitted token"
  [plain-token {:keys [hashed] :as db-tok}]
  (merge db-tok
         (if (active? db-tok)
           (hashers/verify plain-token hashed hasher-options)
           {:valid false :update false})))

(defn valid-token [persistence token-string]
  (m/find-first #(:valid (match-db-token token-string %))
                (user-tokens persistence (:id (token-info token-string)))))

(defrecord DbPATPersistence [db-spec-prov columns joins]
  PATPersistence
  (delete-expired-tokens [this] (delete-expired-tokens! (db-spec-prov) {}))
  (insert-token [this token additional-data]
    (->> (insert-token! (db-spec-prov) (to-insertable token additional-data)) vals first))
  (delete-token [this user-id token-id]
    (if user-id
      (delete-user-token! (db-spec-prov) {:id token-id :user_id user-id})
      (delete-token! (db-spec-prov) {:id token-id})))
  (user-tokens [this id]
    (->> (map-of-db id columns joins)
         (select-tokens-for-user (db-spec-prov))
         (mapv to-readable)))
  (all-tokens [this page]
    (let [pagelen 1000
          offset (* pagelen (or page 0))]
      (->> (map-of-db offset pagelen columns joins)
           (select-all-tokens (db-spec-prov))
           (mapv to-readable))))
  (search-tokens [this user-id page search-term]
    (let [pagelen 1000
          search (when (and search-term (not (str/blank? search-term)))
                   (str "%" search-term "%"))
          offset (* pagelen (or page 0))]
      (->> (search-tokens* (db-spec-prov)
                           (map-of-db offset pagelen columns joins search user-id))
           (mapv to-readable))))
  (update-used-at [this token-id] (update-used-at! (db-spec-prov) {:id token-id})))