(ns codescene.features.repository-provider.gitlab.delta
  "GitLab support for PR Integration

  We use Webhooks to get notified when PR is created or updated.
  Webhooks are maintained when project is deleted, PR integration enabled, disabled,
  and repositories are added or deleted from project. Webhook maintenance is done via REST API,
  and if we lose access we are unable to remove webhooks.

  The webhooks we add have a salted HMAC token generated by us, based on repository external ID.
  GitLab sends these to us when calling the webhook endpoint and we check them. This prevents
  others from creating webhooks (or faking webhook calls) from repositories that never had webhooks
  created by CodeScene. (they can still fake webhook events from repositories that have actual PR integration.

  We subscribe to PR created/updated events, but since those are fired when a PR description is edited and
  other things that shouldn't trigger a delta, we check if we have an existing running or successful delta
  for the PR and commit SHA in question. If we found such a delta job, we ignore the webhook.

  The results are reported as PR comments. The PR comments have a tag to identify which are ours.
  Environments with same token secret will delete each others' comments.
  "
  (:require [clojure.string :as str]
            [codescene.features.client.api :as api-client]
            [codescene.features.delta.pr-status :as pr-status]
            [codescene.features.project.pr-integration :as pi]
            [codescene.features.repository-provider.providers :as providers]
            [codescene.features.security.tokens :as tokens]
            [codescene.features.util.template :as template]
            [codescene.url.url-utils :as url]
            [com.climate.claypoole :as cp]
            [org.clojars.roklenarcic.paginator :as page]
            [taoensso.timbre :as log]
            [codescene.features.delta.file-regions :as file-regions]
            [codescene.features.delta.protocols :as protocols]
            [codescene.features.delta.result.pull-request :as result]
            [codescene.features.repository-provider.gitlab.api :as api]
            [taoensso.timbre :as log]
            [slingshot.slingshot :refer [try+]]))

(defn run-delta?
  "Updating/resolving comments generates more merge request events. We need to make sure to
  only react to specific actions"
  [{{:keys [action state oldrev]} :object_attributes
    {:keys [merge_status]} :changes}]
  ;; PR open
  (or (and (= action "open") (= state "opened"))
      ;; maybe the branch was pushed while PR was closed
      (and (= action "reopen") (= state "opened"))
      ;; pushed branch or target branch switch, not just a MR description change
      (and (= action "update") (= state "opened") (or oldrev merge_status))))

(defn pr-merged?
  [{{:keys [action]} :object_attributes}]
  (= action "merge"))

(def pending-comment-body "# CodeScene MR Check Pending\n\n---\n")

(defn comment-for-pending?
  [comment]
  (str/includes? (:body comment "") pending-comment-body))

(defn ->run-tag [pr-app-id]
  ; identifies specific check's comments by our server's secret
  (format "\n[//]: # (%s)\n" pr-app-id))

(defn to-non-diff-note
  "If you are commenting on a non-diff line then you MUST provide old line and new line
  and old path and new path. Conversely, you MUSTN'T provide old-line when the file is new, or new-line
  when file is deleted. Nasty stuff. So we try to put comment to one of the diff sides, then if errors
  are encountered, we attempt to post as if they are non-diff comments. Basically we're shooting blind
  because we cannot figure out which is the correct form ourselves (as diffs are not guaranteed to be
  there for Cloud)."
  [note]
  (-> note
      (update :new-line #(or % (:old-line note)))
      (update :old-line #(or % (:new-line note)))
      (update :path #(or % (:old-path note)))
      (update :old-path #(or % (:path note)))))

(defn gitlab-project->commit-info [{:keys [id git_http_url]} cloud-only? sha]
  (providers/commit-info
    (assoc (url/repo-url->repo-info git_http_url :gitlab cloud-only?)
      :external-id (api/project-id->gid id))
    sha))

(defn note->simple-str
  [pr-app-id {:keys [old-path new-path old-line new-line message]}]
  (str "* *" (or new-path old-path)
       "*:" (or new-line old-line "")
       " " (str/replace message (->run-tag pr-app-id) "")))

(defn change-item->discussion [pr-app-id {:keys [old-file-name file-name location change-level] :as ctx}]
  (let [{:keys [start after?]} location]
    (merge {:message (str (->run-tag pr-app-id) (template/render-file "templates/delta/file-comment.md" ctx))
            :resolve (= :improvement change-level)}
           (if start
             {(if after? :new-line :old-line) start
              (if after? :path :old-path) (if after? file-name old-file-name)}
             {:new-line 1 :path file-name :old-path old-file-name}))))

(defn pr-comment
  "Interpret delta data into a form closer to the needs of the API calls."
  [{:keys [pr-app-id commit-info pr-id high-risk-active-comment?]} comment status]
  {:body (str (->run-tag pr-app-id) comment)
   :external-id (:external-id commit-info)
   :pr-id pr-id
   :resolved (case status
               (:success :skip-success) true
               :high-risk-success (if high-risk-active-comment? false true)
               (:failure :error :skip-fail :pending) false)})

(defn upsert-pr-comment
  "Updates existing discussion lead comment or creates a new discussion.

  It then sets resolved status. Returns the note."
  [authed-client existing-comment {:keys [external-id pr-id resolved body] :as _pr-comment}]
  (log/debugf "provider=gitlab; action=delta-analysis-results; edit?=%s" (some? existing-comment))
  (let [note (if existing-comment
               (api/edit-merge-request-lead-note authed-client external-id existing-comment body)
               (api/add-merge-request-discussion authed-client external-id pr-id body))]
    (api/resolve-merge-request-lead-note authed-client external-id note resolved)
    note))

(defn delete-comments-except-pending
  "Deletes comments, except the last one, if it is comment about pending analysis.

  Returns the pending comment or nil if there wasn't any."
  [authed-client external-id comments]
  (doseq [comment (butlast comments)]
    (api/delete-merge-request-lead-note authed-client external-id comment))
  (when-let [last-comment (last comments)]
    (if (comment-for-pending? last-comment)
      last-comment
      (do (api/delete-merge-request-lead-note authed-client external-id last-comment)
          nil))))

(defn diff-file-comments
  "Compares previous comments (i.e. existing comments returned by GL API and the comments-to-be-posted)

  Returns a list of comments to delete and a list of comments to actually post."
  [prev-comments notes]
  (let [note->key (fn [{:keys [path old-path new-line old-line message]}]
                    ;; the comment returned by API always has both paths even if we didn't provide both when making them.
                    [new-line old-line (or path old-path) (or old-path path) (str/trim message)])
        comment->key (fn [{:keys [position body]}]
                       (conj (mapv position [:new-line :old-line :new-path :old-path]) (str/trim body)))

        position->prev-comment (zipmap (map comment->key prev-comments) prev-comments)
        new-comment->prev-comment #(some (comp position->prev-comment note->key) [% (to-non-diff-note %)])]
    (reduce
      (fn [ret new-comment]
        ;; if new comment matches one of the prev comments we were going to delete,
        ;; remove the prev comment from set of comments to delete, else add the new comment
        ;; to add list (since old comment list doesn't have it).
        (if-some [prev-comment (new-comment->prev-comment new-comment)]
          (update ret :to-delete disj prev-comment)
          (update ret :to-add conj new-comment)))
      {:to-delete (set prev-comments)
       :to-add []}
      notes)))

(defn get-our-comments [authed-client external-id pr-app-id pr-id]
  (->> (api/get-merge-request-lead-notes authed-client external-id pr-id)
       (filter #(or (str/starts-with? (:body %) (->run-tag pr-app-id))
                    ;; gitlab 11 trims body so the tag we compare to must be trimmed also
                    (str/starts-with? (:body %) (str/trim (->run-tag pr-app-id)))))
       (sort-by :created-at)))

(defn get-our-main-comments
  "Returns our comments that are not a file"
  [authed-client external-id pr-app-id pr-id]
  (remove :position (get-our-comments authed-client external-id pr-app-id pr-id)))

(defn post-file-comments
  "It posts comments to file lines, retrying with a slightly different data model for failed comments.
  It then resolves comments that have to be resolved.

  Old comments are removed if they are not in the set of new comments and new comments are not posted if they already exist.

  Any comments we couldn't post are rendered into a string that is returned."
  [authed-client provider-ref comments comment-contexts]
  (let [{:keys [mr-refs pr-app-id commit-info]} provider-ref
        {:keys [external-id]} commit-info
        notes (map (partial change-item->discussion pr-app-id) comment-contexts)
        {:keys [to-add to-delete]} (diff-file-comments (filter :position comments) notes)
        notes-with-ids (api/add-notes authed-client mr-refs to-add)
        notes-with-ids2 (api/add-notes authed-client mr-refs (map to-non-diff-note (filter :errors notes-with-ids)))]
    (api/resolve-discussions authed-client (concat notes-with-ids notes-with-ids2))
    (doseq [comment to-delete]
      (api/delete-merge-request-lead-note authed-client external-id comment))
    (when-let [notes-with-errors (seq (filter :errors notes-with-ids2))]
      (log/debug "GitLab Notes failed with errors" (pr-str notes-with-errors))
      (str/join \newline
                (conj (map (partial note->simple-str pr-app-id) notes-with-errors)
                      "### Encountered errors posting comments to files, the comments are listed here instead:")))))

(defn create-main-result-comment
  [{:keys [result-options] :as provider-ref} pr-presentable file-comment-errors]
  (let [status (cond
                 (result/negative-review? pr-presentable) :failure
                 (-> pr-presentable :result :high-risk?) :high-risk-success
                 :else :success)]
    (pr-comment provider-ref
                (cond-> (result/presentable->comment pr-presentable :gitlab (not (:discussions? result-options)))
                  file-comment-errors (str \newline file-comment-errors))
                status)))

(defn post-delta-results
  "Posts the results of the delta: the main comment, then any file comments if enabled."
  [delta-authed-client-fn provider-ref pr-presentable]
  (log/debugf "provider=gitlab; action=delta-analysis-results; ref=%s" provider-ref)
  (let [{:keys [commit-info pr-id pr-app-id result-options]} provider-ref
        {:keys [external-id]} commit-info
        result-options (assoc result-options :include-review-findings? (:discussions? result-options))
        authed-client (delta-authed-client-fn provider-ref)
        locator (result/cached-locator #(comp first (file-regions/change-locator % false)))
        comments (get-our-comments authed-client external-id pr-app-id pr-id)
        comment-contexts (when (result/comment? pr-presentable result-options)
                           (result/file-comment-items pr-presentable locator))
        file-comment-errors (when (:discussions? result-options)
                              (post-file-comments authed-client provider-ref comments comment-contexts))
        pending-comment (delete-comments-except-pending authed-client external-id (remove :position comments))]
    (if (result/comment? pr-presentable result-options)
      (->> (create-main-result-comment provider-ref pr-presentable file-comment-errors)
           (upsert-pr-comment authed-client pending-comment))
      (some->> pending-comment (api/delete-merge-request-lead-note authed-client external-id)))))

(defn post-delta-skipped [delta-authed-client-fn provider-ref reason result-url]
  (log/debugf "provider=gitlab; action=delta-analysis-skipped; ref=%s" provider-ref)
  (let [status (if (providers/negative-skip? reason) :skip-fail :skip-success)
        {:keys [commit-info pr-id pr-app-id]} provider-ref
        {:keys [external-id]} commit-info
        authed-client (delta-authed-client-fn provider-ref)
        comments (get-our-main-comments authed-client external-id pr-app-id pr-id)
        pending-comment (delete-comments-except-pending authed-client external-id comments)]
    (if (or (-> provider-ref :result-options :always-comment?) (= :skip-fail status))
      (->> (pr-comment provider-ref
                       (format "### CodeScene MR Check Skipped\n%s\n---\n%s\n[View detailed results in CodeScene](%s)"
                               (result/render-error-title reason)
                               (result/render-error reason)
                               result-url)
                       status)
           (upsert-pr-comment authed-client pending-comment))
      (some->> pending-comment (api/delete-merge-request-lead-note authed-client external-id)))))

(defn post-delta-error [delta-authed-client-fn provider-ref error result-url]
  (log/debugf "provider=gitlab; action=delta-analysis-error; ref=%s" provider-ref)
  (let [{:keys [commit-info pr-id pr-app-id]} provider-ref
        {:keys [external-id]} commit-info
        authed-client (delta-authed-client-fn provider-ref)
        comments (get-our-main-comments authed-client external-id pr-app-id pr-id)
        pending-comment (delete-comments-except-pending authed-client external-id comments)]
    (->> (pr-comment provider-ref
                     (format "### CodeScene MR Check\n%s\n---\n%s\n[View detailed results in CodeScene](%s)"
                             (result/render-error-title error)
                             (result/render-error error)
                             result-url)
                     :error)
         (upsert-pr-comment authed-client pending-comment))))

(defn post-delta-pending
  "Deletes all main comments but the last one, which is edited to show pending status and unresolved.

  This is done so that pending comment doesn't generate an email, if it is the second one."
  [delta-authed-client-fn provider-ref result-url]
  (log/debugf "provider=gitlab; action=delta-analysis-pending; ref=%s" provider-ref)
  (let [authed-client (delta-authed-client-fn provider-ref)]
    (when (-> provider-ref :result-options :show-pending?)
      (->> (pr-comment provider-ref (format "### CodeScene MR Check Pending\n\n---\n[View detailed results in CodeScene](%s)" result-url) :pending)
           (upsert-pr-comment authed-client nil)))))

(defrecord GitlabDelta [delta-authed-client-fn]
  protocols/DeltaResultBoundary
  (delta-pending [this provider-ref result-url]
    (post-delta-pending delta-authed-client-fn provider-ref result-url))
  (delta-results [this provider-ref pr-presentable]
    (post-delta-results delta-authed-client-fn provider-ref pr-presentable))
  (delta-skipped [this provider-ref reason result-url]
    (post-delta-skipped delta-authed-client-fn provider-ref reason result-url))
  (delta-error [this provider-ref error result-url]
    (post-delta-error delta-authed-client-fn provider-ref error result-url)))

(defn webhook-status
  [hooks]
  (cond
    (and (= 1 (count hooks))
         (every? :enabled? hooks))
    :installed
    (zero? (count hooks))
    :not-installed
    :else
    :invalid))

(defn repository-webhook-status
  [pr-integration repositories concurrency]
  (let [our-hook? (pi/our-hook-fn? pr-integration)
        ;; TODO Rok use authed client for each repo separately
        authed-client (pi/-authed-client pr-integration (first repositories))
        hooks (->> (api/get-web-hooks authed-client repositories our-hook? concurrency)
                   (mapv page/unwrap))]
    (mapv #(assoc {} :hooks %
                     :status (webhook-status %)) hooks)))

(defn add-hook [PrIntegrationConfig repo]
  (log/infof "Installing Gitlab webhook to %s" (:url repo))
  (let [secret (cond->> (pi/-hook-secret PrIntegrationConfig)
                 (:external-id repo) (tokens/b64-salted-token (:external-id repo)))]
    [(try+
       (api/add-project-hook
         (pi/-authed-client PrIntegrationConfig repo)
         repo
         (pi/-callback-url PrIntegrationConfig)
         secret)
       (catch [:type :http-error :status 403] e
         (throw
           (api-client/ex-forbidden
             :gitlab
             e
             {:message (str "Cannot add webhook to GitLab Project " (:owner-login repo)
                            "/" (:repo-slug repo) ", you need to be at least a Maintainer.")}))))]))

(defn delete-hook [authed-client repo hook]
  (try+
    (log/infof "Removing Gitlab webhook from %s" (:url hook))
    (api/api-request* authed-client :delete (:url hook) {})
    (catch [:type :http-error :status 403] e
      (throw
        (api-client/ex-forbidden
          :gitlab
          e
          {:message (str "Cannot remove webhook from GitLab Project " (:owner-login repo)
                         "/" (:repo-slug repo) ", you need to be at least a Maintainer.")})))))

(defrecord GitlabHookAccess [concurr-limit]
  protocols/HookAccess
  (-repos-hooks [this PrIntegrationConfig repositories]
    (repository-webhook-status PrIntegrationConfig repositories concurr-limit))
  (-add-hooks [this PrIntegrationConfig repositories]
    (cp/pdoseq concurr-limit
               [repo repositories]
               (add-hook PrIntegrationConfig repo)))
  (-remove-hooks [this PrIntegrationConfig repos]
    (log/info "Uninstalling GitLab Hooks " repos)
    (cp/pdoseq concurr-limit
               [repo repos
                :let [authed-client (pi/-authed-client PrIntegrationConfig repo)]
                hook (:hooks repo)]
               (delete-hook authed-client repo hook))))

(defn mark-merged-pr
  [tx project-id object-attributes]
  (pr-status/delete-open-pr
    tx
    project-id
    (-> (get-in object-attributes [:target :ssh_url])
        url/repo-url->repo-id)
    (str (:iid object-attributes))))