(ns codescene.features.repository-provider.github.delta
  "Provides a gateway to Github checks system, with functions that
  match the CI provider protocols.

  The integration is achieved via a specific Github App for delta analysis.

  The provider ref for this provider is a JSON with:
  - ver, an integer of the version of the provider ref
  - app-installation-id, which is the id of the particular Delta app installation the repository
  of the check.
  - check-run-url, the URL of the API for the particular check run being referenced."
  (:require [clojure.spec.alpha :as s]
            [codescene.cache.core :as cache]
            [codescene.features.delta.file-regions :as file-regions]
            [codescene.features.delta.pr-status :as pr-status]
            [codescene.features.delta.result.pull-request :as result]
            [codescene.features.delta.protocols :as protocols]
            [codescene.features.project.pr-integration :as pi]
            [codescene.features.repository-provider.github.api :as api]
            [codescene.features.repository-provider.github.app :as app]
            [codescene.features.repository-provider.github.checks :as checks]
            [codescene.features.repository-provider.github.review :as review]
            [codescene.features.util.template :as template]
            [codescene.url.url-utils :as url]
            [medley.core :as m]
            [slingshot.slingshot :refer [try+ throw+]]
            [taoensso.timbre :as log]))

(defn annotations [{:keys [filtered-findings]} locator]
  (filtered-findings
    (fn [{:keys [name] :as file-result} finding {:keys [description change-level] :as cd}]
      (let [{:keys [start end]} (locator file-result cd)
            category-description (result/category-description finding)]
        {:path name
         :title (result/title-for finding cd)
         :annotation_level (if (= :warning change-level) "warning" "notice")
         :message (cond-> description
                    category-description (str ". " category-description))
         :start_line (or start 1)
         :end_line (or end 1)}))))

(s/fdef result->checkrun-data
        :args (s/cat :provider-ref map?
                     :result ::result/delta-result-presentable
                     :review-errors? (s/nilable boolean?))
        :ret map?)
(defn result->checkrun-data
  "Generate GitHub Check-run information"
  [provider-ref pr-presentable review-errors?]
  (let [{:keys [file-comments]} (:result-options provider-ref)
        locator (result/cached-locator #(comp file-regions/indices-with-non-empty-region
                                              (file-regions/change-locator % true)))
        text (template/render-file "templates/delta/check-run-text.md"
                                   (merge (result/with-finding-list pr-presentable)
                                          ;; new condition
                                          {:description-details? (not= "review" file-comments)}))
        summary (template/render-file "templates/delta/check-run-summary.md"
                                      (assoc (result/qg-summary->markdown-parts pr-presentable (constantly nil))
                                        :review-errors? review-errors?))]
    {:conclusion (if (-> pr-presentable :qg-summary :fail not-empty) "failure" "success")
     :output {:title "CodeScene PR Check"
              :summary (checks/trim-body summary)
              :text (checks/trim-body text)
              :annotations (if (= "annotations" file-comments) (annotations pr-presentable locator) [])}
     :details_url (:result-url pr-presentable)}))

(defn update-check-run
  [delta-authed-client-fn provider-ref body]
  (let [{:keys [ver app-installation-id check-run-url]} provider-ref]
    (if (= ver 1)
      (checks/update-check-run (delta-authed-client-fn app-installation-id) check-run-url body)
      (log/errorf "provider=github; action=delta-analysis; unknown ref version %s" provider-ref))))

(defn- close-delta
  "Close a check run with a status, url and some text. Used for skipped, error results."
  [delta-authed-client-fn provider-ref status output result-url]
  (log/debugf "provider=github; action=delta-analysis; result=%s; ref=%s" status provider-ref)
  (update-check-run delta-authed-client-fn
                    provider-ref
                    (m/assoc-some {:conclusion status :output output}
                                  :details_url result-url)))

(defn post-delta-success-check-run
  "Close a check run with the results of the delta analysis. Check run is identified by
  the provider ref, specifically check-run-url.

  Report is rendered from a markdown template."
  [delta-authed-client-fn provider-ref pr-presentable review-error?]
  (let [body (result->checkrun-data provider-ref pr-presentable review-error?)]
    (log/debugf "provider=github; action=delta-analysis; result=%s; ref=%s" (:conclusion body) provider-ref)
    (update-check-run delta-authed-client-fn provider-ref body)))

(defn provider-url-fn [comments]
  (let [lookup (reduce (fn [acc {:keys [comment/link file-name category]}]
                         (update acc [category file-name] #(or % link)))
                       {}
                       comments)]
    (fn [category file-name] (get lookup [category file-name]))))

(defn post-delta-success-review
  "Posts a review to PR"
  [delta-authed-client-fn {:keys [ver app-installation-id pull-request-url result-options] :as provider-ref} pr-presentable]
  (when (= "review" (:file-comments result-options))
    (if (= ver 1)
      (let [post? (result/comment? pr-presentable result-options)
            locator (result/cached-locator #(comp file-regions/indices-with-non-empty-region
                                                  (file-regions/change-locator % false)))
            body (fn [comments]
                   (when post?
                     (template/render-file
                       "templates/delta/github-review.md"
                       (result/qg-summary->markdown-parts pr-presentable (provider-url-fn comments)))))]
        (review/new-review
          (delta-authed-client-fn app-installation-id)
          pull-request-url
          (review/review-spec (-> pr-presentable :result :delta-branch-head)
                              (when post? (review/new-code-health-comments pr-presentable locator))
                              "cs-code-health"
                              body)
          (not (result/negative-review? pr-presentable))))
      (log/errorf "provider=github; action=delta-analysis; unknown ref version %s" provider-ref))))

(defn post-delta-skipped
  "Close a check run with the skipped status and the reason keyword."
  [delta-authed-client-fn provider-ref reason result-url]
  (close-delta delta-authed-client-fn
               provider-ref
               "skipped"
               {:title "CodeScene PR Check"
                :summary (format "**Skipped**: %s" (result/render-error reason))}
               result-url))

(defn post-delta-error
  "Close a check run with failure.
  Some delta analysis errors have their messages publicized in the Check Run result."
  [delta-authed-client-fn provider-ref error result-url]
  (close-delta delta-authed-client-fn
               provider-ref
               "failure"
               {:title "CodeScene PR Check"
                :summary (result/render-error-title error)
                :text (result/render-error error)}
               result-url))

(defn post-delta-pending
  "Assumes there is already a pending check run, so it only updates URL."
  [delta-authed-client-fn provider-ref result-url]
  (update-check-run delta-authed-client-fn provider-ref {:details_url result-url}))

(defn- repo-key
  "Here we don't use external ID because onprem doesn't know it"
  [repo-info]
  (select-keys repo-info [:repo-slug :owner-login :host]))

(defn- installed-app-urls
  "To keep cyclomatic complexity happy."
  [authed-client same-owner-repos]
  (log/infof "App selectively installed for %s, looking up repos" (:owner-login (first same-owner-repos)))
  ;; there is no graphql way to query app installations on repos directly, so we detect repository accessibility
  ;; the problem with using this installation detection method is that
  ;; public repos are always accessible, even if app isn't installed. With most customers we can
  ;; expect to have only private repos in project so this is still viable, but we need to revert to old way
  ;; otherwise
  (if (<= (count same-owner-repos) 100)
    (let [repos (->> (api/get-specified-repos authed-client same-owner-repos)
                     (filter :accessible?))]
      (if (every? :private? repos)
        (map :url repos)
        (map :clone_url (app/get-installation-repositories authed-client))))
    (map :clone_url (app/get-installation-repositories authed-client))))

(defn- installed-app-repositories*
  [PrIntegrationConfig same-owner-repos]
  (let [{:keys [owner-login url]} (first same-owner-repos)
        {:keys [repository_selection]} (app/get-user-installation
                                         ;; this is kinda hackish way to request an authed client that is not
                                         ;; an installation client but the general JWT token, the URL is needed
                                         ;; for onprem to find the server
                                         (pi/-authed-client PrIntegrationConfig {:url url})
                                         owner-login)]
    (case repository_selection
      "all" (do (log/infof "App installed generally for %s." owner-login)
                (map :url same-owner-repos))
      "selected" (installed-app-urls (pi/-authed-client PrIntegrationConfig (first same-owner-repos))
                                     same-owner-repos)
      nil)))

(defn- installed-app-repositories
  "Returns a set of maps [repo-slug owner-login host] of repositories that have the app installed
  for the provided user."
  [PrIntegrationConfig same-owner-repos]
  (try+
    (into #{}
          (map #(repo-key (url/repo-url->repo-info % :github false)))
          (installed-app-repositories* PrIntegrationConfig same-owner-repos))
    (catch [:type :http-error :status 401] {:keys [message]}
      (log/debugf "Github app installation check returned unauthorized with message %s, app is not installed"
                  message)
      #{})
    (catch [:type :http-error :status 403] {:keys [message]}
      (log/debugf "Github app installation check returned forbidden with message %s"
                  message)
      (if (and message (re-find #"(?i)suspended" message))
        (with-meta #{} {:error :suspended})
        (throw+)))
    (catch Object _
      (if (= "Unknown PEM object type" (:message &throw-context))
        (with-meta #{} {:error :pem-error})
        (throw+)))))

;; expected fields in provider ref
;; - app-installation-id
;; - check-run-url
(defrecord GithubCheckrunDelta [delta-authed-client-fn]
  protocols/DeltaResultBoundary
  (delta-pending [this provider-ref result-url]
    (post-delta-pending delta-authed-client-fn provider-ref result-url))
  (delta-results [this provider-ref pr-presentable]
    ;; annotations accumulate on checkrun so when replaying results with annotations
    ;; configured we trigger a whole new delta run
    (if (and (:result-replay? provider-ref)
             (= "annotations" (-> provider-ref :result-options :file-comments)))
      (let [{:keys [app-installation-id check-run-url]} provider-ref]
        (checks/rerequest-check-run (delta-authed-client-fn app-installation-id) check-run-url))
      (post-delta-success-check-run
        delta-authed-client-fn
        provider-ref
        pr-presentable
        (try
          (post-delta-success-review delta-authed-client-fn provider-ref pr-presentable)
          false
          (catch Exception e
            (log/error e "Failed to post GitHub Review")
            true)))))
  (delta-skipped [this provider-ref reason result-url]
    (post-delta-skipped delta-authed-client-fn provider-ref reason result-url))
  (delta-error [this provider-ref error result-url]
    (post-delta-error delta-authed-client-fn provider-ref error result-url)))

(defrecord GithubAppCI []
  protocols/CIProvider
  (ci-installed? [this PrIntegrationConfig repositories]
    (let [installed-repo-map (update-vals (group-by :owner-login repositories) #(installed-app-repositories PrIntegrationConfig %))]
      (mapv
        (fn [repo]
          (let [repo (update repo :host #(or % "github.com"))
                installed (installed-repo-map (:owner-login repo))]
            (assoc repo :ci-installed? (contains? installed (repo-key repo))
                        :ci-status-string (case (-> installed meta :error)
                                            :suspended "GitHub App SUSPENDED"
                                            :pem-error "GitHub App Private Key error"
                                            nil))))
        repositories)))
  (enable-ci-integration! [this PrIntegrationConfig repositories])
  (disable-ci-integration! [this PrIntegrationConfig repositories]))

(defn pr-for-external-id
  "Returns the pull request referenced by the external ID."
  [authed-client external-id]
  (when-let [[_ typ url] (re-find #"(PR|COV)#(.+)" (or external-id ""))]
    (try+
      [typ (api/api-request authed-client :get url {})]
      (catch [:type :http-error :status 403] _
        ;; no PR found
        ))))

(cache/memo-scoped #'pr-for-external-id)

(defn create-check-run
  [checkrun-authed-client hook-body pull-request]
  (checks/create-check-run checkrun-authed-client
                           (->> hook-body :repository :url)
                           {:name (->> pull-request :base :ref (format "CodeScene Code Health Review (%s)"))
                            :head-commit-sha (->> pull-request :head :sha)
                            :external-id (->> pull-request :url (str "PR#"))}))

(defn pr-authors [authed-client pull-request]
  (let [url (-> pull-request :user :url)
        {:keys [login name]} (api/api-request authed-client :get url {})]
    [login name]))

(defn mark-merged-pr
  [tx project-id body]
  (pr-status/delete-open-pr
    tx
    project-id
    (url/repo-url->repo-id (-> body :repository :git_url))
    (-> body :pull_request :number str)))