(ns codescene.features.repository-provider.bitbucket.delta
  "Utilities for publishing Delta results (successful or failures).

  The result is provided to user in shape of a Pull Request comment (with any previous comment
  being deleted as to not accumulate comments on PR) and an more visual, platform based, indicator.

  This indicator is commit based, made on HEAD of PR branch. It can be either a Build Status or Code Insights report.

  Neither offer much space, with report offering at least some space. You can gate merging of PR on build
  status passing, but you cannot do on report. BB cannot be configured to allow merge on some Builds passing and
  not the rest, so customer has some other Build they want as merge gate, but not CodeScene, then they should
  switch CodeScene to Report.

  Functions with 'build' in name deal with builds, those with 'report' deal with reports.
  Which form of results is configured is derived from presence of keys in provider-ref:
  - link presence = we need to manage build
  - commit-info present = we need to manage report"
  (:require [codescene.features.delta.protocols :as protocols]
            [codescene.features.project.pr-integration :as pi]
            [codescene.features.repository-provider.bitbucket.result :as result]
            [codescene.features.security.tokens :as tokens]
            [codescene.features.repository-provider.providers :as providers]
            [codescene.features.repository-provider.bitbucket.api :as api]
            [codescene.url.url-utils :as url]
            [com.climate.claypoole :as cp]
            [medley.core :as m]
            [org.clojars.roklenarcic.paginator :as page]
            [taoensso.timbre :as log]))

(defn pr-app-id
  "Creates an ID for PR. The ID is combined repo ID + base branch + instance secret.

  The per-app uniqueness should at least in theory already stem from app-key, which at least
  in Cloud should be unique by default. The secret can be the stack's token secret, and this is
  used to turn other arguments into a nice base64 byte soup that should be unique in regard to parameters
  above. If server AND app-key are same for two servers then they will delete each other's comments
  on the same PR, so the secret is not terribly important i.e. failure mode is not any sort of
  security problem."
  [app-key external-id base-ref secret]
  (tokens/b64-token (str app-key "-" external-id "-" base-ref) secret))

(defn post-comment [commit-info->client
                    {:keys [delete-comments-with-runtag add-pull-request-comment]}
                    provider-ref
                    comment]
  (let [{:keys [pr-id pr-app-id]} provider-ref
        run-tag (format "\n[//]: # (%s)\n" pr-app-id)
        client (-> provider-ref :target-commit-info commit-info->client)
        ;; due to forking logic we need access to PR target repository to post pull-request comment
        repo-info (:target-commit-info provider-ref)]
    (log/debugf "deleting old comments, provider=bitbucket; action=delta-analysis; ref=%s" provider-ref)
    (delete-comments-with-runtag client repo-info pr-id run-tag)
    (when comment
      (log/debugf "post comment, provider=bitbucket; action=delta-analysis; ref=%s" provider-ref)
      (add-pull-request-comment client repo-info pr-id (str run-tag comment)))))

(defn update-build
  "Updates build"
  [commit-info->client provider-ref {:keys [state text result-url]}]
  (let [{:keys [link commit-info]} provider-ref]
    (when link
      (log/debugf "update build, provider=bitbucket; action=delta-analysis; ref=%s" provider-ref)
      (api/update-commit-build-by-url
        (commit-info->client commit-info)
        link
        (m/assoc-some {:state state} :description text :url result-url)))))

(defn update-report
  "Update Code Insights Report if it exists, by first loading the report then
  updating it and using PUT to store it."
  [commit-info->client provider-ref report-updates]
  (let [{:keys [pr-app-id result-options commit-info]} provider-ref
        report-id (str "codescene-" pr-app-id)
        authed-client (commit-info->client commit-info)
        {:keys [report? annotations?]} result-options]
    (when-let [report (and report? (api/get-commit-report authed-client commit-info report-id))]
      (log/debugf "Update report, provider=bitbucket; action=delta-analysis; ref=%s" provider-ref)
      (log/debugf "Report annotations, provider=bitbucket; enabled=%s, #=%s" annotations? (count (:annotations report-updates)))
      (doseq [annotation-batch (partition-all 100 (take 1000 (:annotations report-updates)))
              :when annotations?]
        (api/add-annotations authed-client commit-info report-id annotation-batch))
      (api/put-commit-report
        authed-client
        commit-info
        (merge {:id report-id} report (dissoc report-updates :annotations))
        report-id)
      report-id)))

(defn post-delta [commit-info->client provider-ref {:keys [comment build report]}]
  (post-comment commit-info->client
                {:add-pull-request-comment api/add-pull-request-comment
                 :delete-comments-with-runtag api/delete-comments-with-runtag}
                provider-ref
                comment)
  (update-build commit-info->client provider-ref build)
  (update-report commit-info->client provider-ref report))

(defn provider-ref-add-pending-build
  "Add a pending build to PR and save link into provider-ref, requires access to
  source repository."
  [{:keys [pr-app-id pr-id target-branch commit-info result-options] :as provider-ref} commit-info->client details-url]
  (let [resp (when (:build? result-options)
               (api/create-commit-build
                 (commit-info->client commit-info)
                 commit-info
                 {:name (format "CodeScene (%s) PR#%s" target-branch pr-id)
                  :url details-url
                  :key (cond-> pr-app-id (< 40 (count pr-app-id)) (subs 0 40))
                  :state "INPROGRESS"}))]
    (m/assoc-some provider-ref :link (get-in resp [:links :self :href]))))

(defn provider-ref-add-pending-report
  "Add a pending build to PR and save link into provider-ref, requires access to
  source repository."
  [{:keys [pr-app-id commit-info result-options target-branch] :as provider-ref} commit-info->client]
  (when (:report? result-options)
    (log/debugf "provider=bitbucket; action=delta-analysis; create build report pending; commit-info=%s"
                commit-info)
    (api/put-commit-report
      (commit-info->client commit-info)
      commit-info
      {:id (str "codescene-" pr-app-id)
       :title (format "CodeScene (%s)" target-branch)
       :details "CodeScene Delta Analysis"
       :report_type "TEST"
       :reporter "CodeScene"
       :logo_url "https://codescene.io/cs-favicon.png"
       :result "PENDING"}
      (str "codescene-" pr-app-id)))
  provider-ref)

(defn pr-endpoint
  "Parses PR endpoint and provider-def (:source or :destination) for branch, access token, commit-info, repo-name"
  [pr-endpoint access cloud?]
  (let [repository-url (-> pr-endpoint :repository :links :html :href (str ".git"))
        commit-info (-> repository-url
                        (url/repo-url->repo-info :bitbucket cloud?)
                        (providers/commit-info (get-in pr-endpoint [:commit :hash]))
                        (assoc :external-id (-> pr-endpoint :repository :uuid)))]
    {:commit-info commit-info
     :repository-url repository-url
     :branch (some-> (get-in pr-endpoint [:branch :name]) providers/branch-ref->branch-name)
     :access access
     :repo-name (get-in pr-endpoint [:repository :full_name])}))

(defn ->provider-ref
  "Create provider ref without report and build."
  [source target pr-id pr-app-id result-options]
  {:pr-id pr-id
   ; we use source's tenant to act on build statuses but target tenant and target
   ; repository url to edit pull request comments
   :commit-info (assoc (:commit-info source) :access (:access source))
   :pr-app-id pr-app-id
   :result-options result-options
   :target-branch (:branch target)
   ; since we got the event, the target repo has app installed
   :target-commit-info (assoc (:commit-info target) :access (:access target))})

(defrecord BitbucketDelta [commit-info->client]
  protocols/DeltaResultBoundary
  (delta-pending [this provider-ref result-url]
    (post-delta commit-info->client provider-ref (result/pending-update result-url)))
  (delta-results [this provider-ref pr-presentable]
    (->> (result/result-update pr-presentable result/change-item->annotation (:result-options provider-ref))
         (post-delta commit-info->client provider-ref)))
  (delta-skipped [this provider-ref reason result-url]
    (post-delta commit-info->client provider-ref (result/skipped-update provider-ref reason result-url)))
  (delta-error [this provider-ref error result-url]
    (post-delta commit-info->client provider-ref (result/error-update error result-url))))

(defn- enabled-hook? [{:keys [events enabled?]}]
  (and enabled? (every? (set events) ["pullrequest:created"
                                      "pullrequest:updated"
                                      "pullrequest:fulfilled"])))

(defn webhook-status
  [pr-integration hooks]
  (cond
    (and (= 1 (count hooks))
         (every? enabled-hook? hooks)
         #_(every? #(pi/hook-token-verify pr-integration %) hooks))
    :installed
    (zero? (count hooks))
    :not-installed
    :else
    :invalid))

(defn repository-webhook-status
  [pr-integration repositories concurrency]
  (let [our-hook? (pi/our-hook-fn? pr-integration)
        ;; TODO Rok use authed client for each repo separately
        authed-client (pi/-authed-client pr-integration (first repositories))
        hooks (->> (api/get-repos-webhooks authed-client repositories our-hook? concurrency)
                   (mapv page/unwrap))]
    (mapv #(assoc {} :hooks %
                     :status (webhook-status pr-integration %)) hooks)))

(defrecord BitbucketHookAccess [concurr-limit]
  protocols/HookAccess
  (-repos-hooks [this PrIntegrationConfig repositories]
    (repository-webhook-status PrIntegrationConfig repositories concurr-limit))
  (-add-hooks [this PrIntegrationConfig repositories]
    (cp/pdoseq concurr-limit
               [repo repositories]
               (log/infof "Installing Bitbucket webhook to %s" (:url repo))
               (api/add-webhook
                 (pi/-authed-client PrIntegrationConfig repo)
                 repo
                 (pi/-callback-url PrIntegrationConfig)
                 ["pullrequest:created"
                  "pullrequest:updated"
                  "pullrequest:fulfilled"])))
  (-remove-hooks [this PrIntegrationConfig repos]
    (cp/pdoseq concurr-limit
               [repo repos
                :let [authed-client (pi/-authed-client PrIntegrationConfig repo)]
                hook (:hooks repo)]
               (log/infof "Removing Bitbucket webhook from %s" (:url hook))
               (api/api-request* authed-client :delete (:url hook) {}))))